/*****************************************************************************
 * Copyright (c) 2020, 2023, 2025 CEA LIST and others.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Bug 581523
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Issue GL-9
 *****************************************************************************/

package org.eclipse.papyrus.model2doc.integration.emf.documentstructuretemplate.representation.architecture.commands;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.transaction.TransactionalEditingDomain;
import org.eclipse.papyrus.model2doc.core.generatorconfiguration.IDocumentStructureGeneratorConfiguration;
import org.eclipse.papyrus.model2doc.core.logger.IModel2DocLogger;
import org.eclipse.papyrus.model2doc.emf.documentstructuretemplate.DocumentTemplate;
import org.eclipse.papyrus.model2doc.emf.documentstructuretemplate.DocumentTemplatePrototype;
import org.eclipse.papyrus.model2doc.emf.documentstructuretemplate.TextDocumentTemplate;

/**
 * Create a DocumentTemplate Editor view
 */
public class CreateDocumentTemplateEditorViewCommand extends AbstractCreatePapyrusEditorViewCommand<DocumentTemplate> {

	/**
	 * the {@link DocumentTemplatePrototype} used to create the {@link DocumentTemplate} model and its editor view
	 */
	private final DocumentTemplatePrototype prototype;

	/**
	 * the main title of the created {@link DocumentTemplate}
	 */
	private final String mainTitle;

	/**
	 *
	 * Constructor.
	 *
	 * @param domain
	 * @param documentTemplatePrototype
	 * @param documentName
	 * @param documentMainTitle
	 * @param semanticContext
	 * @param graphicalContext
	 * @param openAfterCreation
	 * @param logger
	 */
	public CreateDocumentTemplateEditorViewCommand(final TransactionalEditingDomain domain, final DocumentTemplatePrototype documentTemplatePrototype, final String documentName, final String documentMainTitle, final EObject semanticContext,
			final EObject graphicalContext, final boolean openAfterCreation, final IModel2DocLogger logger) {
		super(domain, "Create new DocumentTemplate", documentName, semanticContext, graphicalContext, openAfterCreation, logger); //$NON-NLS-1$
		this.prototype = documentTemplatePrototype;
		this.mainTitle = documentMainTitle;
	}

	/**
	 *
	 * Constructor.
	 *
	 * @param domain
	 * @param documentTemplatePrototype
	 * @param documentName
	 * @param documentMainTitle
	 * @param semanticContext
	 * @param openAfterCreation
	 * @param logger
	 */
	public CreateDocumentTemplateEditorViewCommand(final TransactionalEditingDomain domain, final DocumentTemplatePrototype documentTemplatePrototype, final String documentName, final String documentMainTitle, final EObject semanticContext,
			final boolean openAfterCreation, final IModel2DocLogger logger) {
		this(domain, documentTemplatePrototype, documentName, documentMainTitle, semanticContext, null, openAfterCreation, logger);
	}

	/**
	 *
	 * @see org.eclipse.emf.transaction.RecordingCommand#doExecute()
	 *
	 */
	@Override
	protected void doExecute() {
		final DocumentTemplate template = this.prototype.getDocumentTemplate();

		final DocumentTemplate newInstance = EcoreUtil.copy(template);

		if (newInstance instanceof TextDocumentTemplate) {
			((TextDocumentTemplate) newInstance).setMainTitle(this.mainTitle);
		}

		if (this.graphicalContext != null) {// graphical context, see bug 581523
			attachToResource(this.graphicalContext, newInstance);
		} else {
			attachToResource(this.semanticContext, newInstance);
		}

		final IDocumentStructureGeneratorConfiguration generator = newInstance.getDocumentStructureGeneratorConfiguration();
		if (null != generator) {
			generator.setDocumentName(this.editorViewName);
		}

		newInstance.setDocumentTemplatePrototype(this.prototype);
		newInstance.setGraphicalContext(this.graphicalContext);
		newInstance.setSemanticContext(this.semanticContext);
		newInstance.setName(this.editorViewName);
		if (this.openAfterCreation) {
			openEditor(newInstance);
		}
		if (newInstance.eResource() != null) {
			// we suppose all is ok
			this.createdEditorView = newInstance;
		}

	}

}
