/*****************************************************************************
 * Copyright (c) 2019 CEA LIST.
 *
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *  Vincent Lorenzo (CEA LIST) vincent.lorenzo@cea.fr - Initial API and implementation
 *
 *****************************************************************************/


package org.eclipse.papyrus.cdo.security.handlers;

import org.eclipse.core.commands.AbstractHandler;
import org.eclipse.core.commands.ExecutionEvent;
import org.eclipse.core.commands.ExecutionException;
import org.eclipse.emf.cdo.security.Directory;
import org.eclipse.emf.cdo.security.Group;
import org.eclipse.emf.cdo.security.Realm;
import org.eclipse.emf.cdo.security.Role;
import org.eclipse.emf.cdo.security.SecurityFactory;
import org.eclipse.emf.cdo.security.User;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.papyrus.cdo.security.internal.ui.CDOUserDialog;
import org.eclipse.papyrus.infra.ui.util.SelectionHelper;
import org.eclipse.swt.widgets.Display;

public class CreateUserHandler extends AbstractHandler {

	/**
	 * The name of the role allowing to declare the home for the user
	 */
	private static final String HOME_FOLDER_ROLE_NAME = "Home Folder /home"; //$NON-NLS-1$

	/**
	 * if <code>true</code>, a dialog will be open to allow to the administrator to edit the new created user
	 */
	protected final boolean editUserAfterRegistration;

	/**
	 *
	 * Constructor.
	 *
	 */
	public CreateUserHandler() {
		this(true);
	}

	/**
	 *
	 * Constructor.
	 *
	 * @param editUserAfterCreation
	 *            if <code>true</code>, a dialog will be open to allow to the administrator to edit the new created user
	 *
	 */
	public CreateUserHandler(boolean editUserAfterCreation) {
		this.editUserAfterRegistration = editUserAfterCreation;
	}

	/**
	 *
	 * @see org.eclipse.core.commands.IHandler#execute(org.eclipse.core.commands.ExecutionEvent)
	 *
	 * @param event
	 * @return
	 * @throws ExecutionException
	 */
	@Override
	public Object execute(final ExecutionEvent event) throws ExecutionException {
		// 1. create a new user
		final User user = SecurityFactory.eINSTANCE.createUser();

		// 2. register the user
		registerUser(user);

		return null;
	}

	/**
	 * Register the user and open a dialog to edit it, according to the boolean field {@link #editUserAfterRegistration}
	 *
	 * @param user
	 *            the user to register
	 */
	protected void registerUser(final User user) {
		// 1. get the root of the file
		final Realm realm = getRealm();

		// 2. register user
		attachUserToUserGroup(realm, user);

		// 3. init user
		initUser(realm, user);

		if (this.editUserAfterRegistration) {
			// 4. edit user
			final CDOUserDialog d = new CDOUserDialog(Display.getDefault().getActiveShell(), user);
			d.open();
		}
	}


	/**
	 *
	 * This method attach the user to the user group
	 *
	 * @param realm
	 * @param user
	 */
	protected void attachUserToUserGroup(final Realm realm, final User user) {
		final Directory userDirectory = realm.getDefaultUserDirectory();
		userDirectory.getItems().add(user);
	}

	/**
	 * This method init the user
	 *
	 * @param realm
	 * @param user
	 */
	protected final void initUser(final Realm realm, final User user) {
		final Directory roleDirectory = realm.getDefaultRoleDirectory();
		final Role homeFolder = roleDirectory.getRole(HOME_FOLDER_ROLE_NAME);
		final Role allObjectsReader = roleDirectory.getRole(Role.ALL_OBJECTS_READER);
		final Role resourceTreeReaderRole = roleDirectory.getRole(Role.RESOURCE_TREE_READER);
		final Group userGroup = realm.getGroup("Users"); //$NON-NLS-1$
		userGroup.getUsers().add(user);
		user.getRoles().add(homeFolder);
		user.getRoles().add(allObjectsReader);
		user.getRoles().add(resourceTreeReaderRole);

	}

	/**
	 *
	 * @return
	 *         the realm (which is the root of the security file)
	 */
	protected static final Realm getRealm() {
		final IStructuredSelection service = SelectionHelper.getCurrentStructuredSelection();
		Object first = service.getFirstElement();
		if (first instanceof Realm) {
			return ((Realm) first).getRealm();
		}
		if (first instanceof Directory) {
			return ((Directory) first).getRealm();
		}
		if (first instanceof User) {
			return ((User) first).getRealm();
		}
		return null;
	}

	/**
	 *
	 * @see org.eclipse.core.commands.AbstractHandler#setEnabled(java.lang.Object)
	 *
	 * @param evaluationContext
	 */
	@Override
	public void setEnabled(Object evaluationContext) {
		setBaseEnabled(allowCreation());
	}

	/**
	 *
	 * @return
	 *         <code>true</code>if all required elements are available for a user creation
	 */
	public static boolean allowCreation() {
		final Realm realm = getRealm();
		if (null == realm) {
			return false;
		}
		// check the security file contains expected field
		if (null == realm.getDefaultGroupDirectory()) {
			return false;
		}
		;
		final Directory roleDirectory = realm.getDefaultRoleDirectory();
		if (null == roleDirectory) {
			return false;
		}
		if (null == roleDirectory.getRole(HOME_FOLDER_ROLE_NAME)) {
			return false;
		}
		if (null == roleDirectory.getRole(Role.ALL_OBJECTS_READER)) {
			return false;
		}
		if (null == roleDirectory.getRole(Role.RESOURCE_TREE_READER)) {
			return false;
		}

		return true;

	}

}
