/*****************************************************************************
 * Copyright (c) 2013, 2017 CEA LIST.
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 *
 * Contributors:
 *   CEA LIST - Initial API and implementation
 *****************************************************************************/
package org.eclipse.papyrus.cdo.internal.ui.wizards;

import java.lang.reflect.InvocationTargetException;
import java.util.function.Predicate;

import org.eclipse.core.runtime.IProgressMonitor;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.core.runtime.SubMonitor;
import org.eclipse.emf.cdo.explorer.CDOExplorerUtil;
import org.eclipse.emf.cdo.explorer.checkouts.CDOCheckout;
import org.eclipse.emf.cdo.explorer.checkouts.CDOCheckoutManager;
import org.eclipse.emf.cdo.explorer.ui.checkouts.CDOCheckoutContentProvider;
import org.eclipse.jface.layout.GridDataFactory;
import org.eclipse.jface.operation.IRunnableContext;
import org.eclipse.jface.operation.IRunnableWithProgress;
import org.eclipse.jface.viewers.ISelectionChangedListener;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.SelectionChangedEvent;
import org.eclipse.jface.viewers.StructuredSelection;
import org.eclipse.jface.viewers.TreeViewer;
import org.eclipse.osgi.util.NLS;
import org.eclipse.papyrus.cdo.ui.Activator;
import org.eclipse.papyrus.cdo.internal.ui.l10n.Messages;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Control;
import org.eclipse.ui.statushandlers.StatusManager;

import com.google.common.base.Supplier;
import com.google.common.eventbus.EventBus;

/**
 * This is the CheckoutSelectionBlock type. Enjoy.
 */
public class CheckoutSelectionBlock {

	private final EventBus bus;

	private final Supplier<? extends IRunnableContext> runnableContext;

	private TreeViewer checkoutList;

	private CDOCheckout selectedCheckout;


	public CheckoutSelectionBlock(EventBus bus, Supplier<? extends IRunnableContext> runnableContext) {
		this.bus = bus;
		this.runnableContext = runnableContext;
	}

	@SuppressWarnings("restriction")
	public Control createControl(Composite parent) {
		checkoutList = CDOCheckoutContentProvider.createTreeViewer(parent, new Predicate<Object>() {
			@Override
			public boolean test(Object child) {
				if (child instanceof CDOCheckout) {
					CDOCheckout checkout = (CDOCheckout) child;
					return !checkout.isReadOnly();
				}
				return false;
			}
		});

		GridDataFactory.fillDefaults().grab(true, true).applyTo(checkoutList.getControl());

		if (selectedCheckout != null) {
			checkoutList.setSelection(new StructuredSelection(selectedCheckout));
			selected(selectedCheckout);
		}

		checkoutList.addSelectionChangedListener(new ISelectionChangedListener() {

			@Override
			public void selectionChanged(SelectionChangedEvent event) {
				IStructuredSelection sel = (IStructuredSelection) event.getSelection();

				if (sel.isEmpty()) {
					if (selectedCheckout != null) {
						// veto empty selection
						checkoutList.setSelection(new StructuredSelection(selectedCheckout));
					}
				} else {
					selected((CDOCheckout) sel.getFirstElement());
				}
			}
		});

		// initially select the first connected repo
		for (CDOCheckout next : CDOExplorerUtil.getCheckoutManager().getCheckouts()) {

			if (next.isOpen()) {
				selected(next);
				checkoutList.setSelection(new StructuredSelection(next));
				break;
			}
		}

		return checkoutList.getControl();
	}

	public void dispose() {
		// pass
	}

	public void setEnabled(boolean enabled) {
		if (checkoutList != null) {
			checkoutList.getControl().setEnabled(enabled);
		}
	}

	public boolean isEnabled() {
		return (checkoutList != null) && checkoutList.getControl().isEnabled();
	}

	void selected(final CDOCheckout checkout) {
		selectedCheckout = checkout;

		if (!checkout.isOpen()) {
			try {
				runnableContext.get().run(true, false, new IRunnableWithProgress() {

					@Override
					public void run(IProgressMonitor monitor) throws InvocationTargetException, InterruptedException {

						SubMonitor sub = SubMonitor.convert(monitor, NLS.bind(Messages.CheckoutSelectionBlock_0, checkout.getLabel()), IProgressMonitor.UNKNOWN);

						try {
							checkout.open();

							// yes, it's a busy wait, but there's not much
							// to be done about that.
							final long deadline = System.currentTimeMillis() + 5000L;
							while (!checkout.isOpen()) {
								Thread.sleep(250L);
								if (System.currentTimeMillis() >= deadline) {
									break;
								}
							}

							checkoutList.getControl().getDisplay().asyncExec(new Runnable() {

								@Override
								public void run() {
									if (!checkoutList.getControl().isDisposed()) {
										checkoutList.update(checkout, null);
										bus.post(checkout);
										selectionChanged();
									}
								}
							});
						} finally {
							sub.done();
						}
					}
				});
			} catch (Exception e) {
				StatusManager.getManager().handle(new Status(IStatus.ERROR, Activator.PLUGIN_ID, Messages.CheckoutSelectionBlock_1, e), StatusManager.SHOW);
			}
		}

		bus.post(checkout);
		selectionChanged();
	}

	protected void selectionChanged() {
	}

	public void setSelectedCheckout(CDOCheckout checkout) {
		this.selectedCheckout = checkout;

		if (checkoutList != null) {
			checkoutList.setSelection(new StructuredSelection(selectedCheckout));
			selected(selectedCheckout);
		}
	}

	public CDOCheckout getSelectedCheckout() {
		return selectedCheckout;
	}
}
