/**
 * Copyright (c) 2018 CEA LIST.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *     Shuai Li (CEA LIST) <shuai.li@cea.fr> - initial API and implementation
 */
package org.eclipse.papyrus.designer.transformation.languages.cpp.library.statemachine;

import java.util.Objects;
import java.util.function.Consumer;
import org.eclipse.papyrus.designer.languages.cpp.profile.C_Cpp.Include;
import org.eclipse.papyrus.designer.monitoring.prefs.MonitoringPreferencesUtil;
import org.eclipse.papyrus.designer.transformation.languages.cpp.library.IncludeUtils;
import org.eclipse.papyrus.designer.transformation.profile.Transformation.Monitored;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.StateMachine;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class MonitoringTransformation {
  public static final String MONITOR_SOCKET_NAME = "monitorSocket";

  public static final String MONITOR_ADDRESS_NAME = "monitorAddress";

  private SM2ClassesTransformationCore core;

  private org.eclipse.uml2.uml.Class superContext;

  private boolean isMonitored;

  public MonitoringTransformation(final SM2ClassesTransformationCore core) {
    this.core = core;
    this.superContext = core.superContext;
    this.isMonitored = MonitoringPreferencesUtil.getEnabled();
  }

  public void appendInclude() {
    if (this.isMonitored) {
      final Include include = UMLUtil.<Include>getStereotypeApplication(this.superContext, Include.class);
      if ((include != null)) {
        final String header = include.getHeader();
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("#include <netinet/in.h>");
        boolean _contains = header.contains(_builder);
        boolean _not = (!_contains);
        if (_not) {
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("#include <netinet/in.h>");
          IncludeUtils.appendIncludeHeader(this.superContext, _builder_1.toString());
        }
      }
      if ((include != null)) {
        final String body = UMLUtil.<Include>getStereotypeApplication(this.superContext, Include.class).getHeader();
        StringConcatenation _builder_2 = new StringConcatenation();
        _builder_2.append("#include <unistd.h>");
        boolean _contains_1 = body.contains(_builder_2);
        boolean _not_1 = (!_contains_1);
        if (_not_1) {
          StringConcatenation _builder_3 = new StringConcatenation();
          _builder_3.append("#include <unistd.h>");
          IncludeUtils.appendIncludeBody(this.superContext, _builder_3.toString());
        }
        StringConcatenation _builder_4 = new StringConcatenation();
        _builder_4.append("#include <sys/socket.h>");
        boolean _contains_2 = body.contains(_builder_4);
        boolean _not_2 = (!_contains_2);
        if (_not_2) {
          StringConcatenation _builder_5 = new StringConcatenation();
          _builder_5.append("#include <sys/socket.h>");
          IncludeUtils.appendIncludeBody(this.superContext, _builder_5.toString());
        }
        StringConcatenation _builder_6 = new StringConcatenation();
        _builder_6.append("#include <arpa/inet.h>");
        boolean _contains_3 = body.contains(_builder_6);
        boolean _not_3 = (!_contains_3);
        if (_not_3) {
          StringConcatenation _builder_7 = new StringConcatenation();
          _builder_7.append("#include <arpa/inet.h>");
          IncludeUtils.appendIncludeBody(this.superContext, _builder_7.toString());
        }
      }
    }
  }

  public Property createMonitorAttributes() {
    Property _xifexpression = null;
    if (this.isMonitored) {
      Property _xblockexpression = null;
      {
        this.superContext.createOwnedAttribute(MonitoringTransformation.MONITOR_SOCKET_NAME, this.core.intType);
        _xblockexpression = this.superContext.createOwnedAttribute(MonitoringTransformation.MONITOR_ADDRESS_NAME, this.core.sockAddrInType);
      }
      _xifexpression = _xblockexpression;
    }
    return _xifexpression;
  }

  public void createDestructor() {
    if (this.isMonitored) {
      final Function1<Operation, Boolean> _function = (Operation it) -> {
        return Boolean.valueOf((StereotypeUtil.isApplied(it, "StandardProfile::Destroy") && Objects.equals(it.getName(), this.superContext.getName())));
      };
      Iterable<Operation> sourceDestructors = IterableExtensions.<Operation>filter(this.superContext.getOwnedOperations(), _function);
      final Function1<Operation, Operation> _function_1 = (Operation it) -> {
        return it;
      };
      Iterable<Operation> targetDestructors = IterableExtensions.<Operation, Operation>map(sourceDestructors, _function_1);
      boolean _isEmpty = IterableExtensions.isEmpty(targetDestructors);
      if (_isEmpty) {
        Operation destructor = this.superContext.createOwnedOperation(this.superContext.getName(), null, null);
        StereotypeUtil.apply(destructor, "StandardProfile::Destroy");
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("close(");
        _builder.append(MonitoringTransformation.MONITOR_SOCKET_NAME);
        _builder.append(");");
        this.core.createOpaqueBehavior(this.superContext, destructor, _builder.toString());
      } else {
        final Consumer<Operation> _function_2 = (Operation it) -> {
          Behavior _head = IterableExtensions.<Behavior>head(it.getMethods());
          OpaqueBehavior opaque = ((OpaqueBehavior) _head);
          if ((opaque != null)) {
            String body = opaque.getBodies().get(0);
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append(body);
            _builder_1.newLineIfNotEmpty();
            _builder_1.append("close(");
            _builder_1.append(MonitoringTransformation.MONITOR_SOCKET_NAME);
            _builder_1.append(");");
            this.core.createOpaqueBehavior(this.superContext, it, _builder_1.toString());
          } else {
            StringConcatenation _builder_2 = new StringConcatenation();
            _builder_2.append("close(");
            _builder_2.append(MonitoringTransformation.MONITOR_SOCKET_NAME);
            _builder_2.append(");");
            this.core.createOpaqueBehavior(this.superContext, it, _builder_2.toString());
          }
        };
        targetDestructors.forEach(_function_2);
      }
    }
  }

  public void createConstructor() {
    if (this.isMonitored) {
      final Function1<Operation, Boolean> _function = (Operation it) -> {
        return Boolean.valueOf((StereotypeUtil.isApplied(it, "StandardProfile::Create") && Objects.equals(it.getName(), this.superContext.getName())));
      };
      Iterable<Operation> sourceContructors = IterableExtensions.<Operation>filter(this.superContext.getOwnedOperations(), _function);
      final Function1<Operation, Operation> _function_1 = (Operation it) -> {
        return it;
      };
      Iterable<Operation> targetContructors = IterableExtensions.<Operation, Operation>map(sourceContructors, _function_1);
      boolean _isEmpty = IterableExtensions.isEmpty(targetContructors);
      if (_isEmpty) {
        Operation constructor = this.superContext.createOwnedOperation(this.superContext.getName(), null, null);
        StereotypeUtil.apply(constructor, "StandardProfile::Create");
        StringConcatenation _builder = new StringConcatenation();
        _builder.append(MonitoringTransformation.MONITOR_SOCKET_NAME);
        _builder.append(" = socket(AF_INET, SOCK_DGRAM, 17);");
        _builder.newLineIfNotEmpty();
        _builder.append(MonitoringTransformation.MONITOR_ADDRESS_NAME);
        _builder.append(".sin_addr.s_addr = inet_addr(\"");
        String _ipAddress = MonitoringPreferencesUtil.getIpAddress();
        _builder.append(_ipAddress);
        _builder.append("\");");
        _builder.newLineIfNotEmpty();
        _builder.append(MonitoringTransformation.MONITOR_ADDRESS_NAME);
        _builder.append(".sin_family = AF_INET;");
        _builder.newLineIfNotEmpty();
        _builder.append(MonitoringTransformation.MONITOR_ADDRESS_NAME);
        _builder.append(".sin_port = htons(");
        int _port = MonitoringPreferencesUtil.getPort();
        _builder.append(_port);
        _builder.append(");");
        this.core.createOpaqueBehavior(this.superContext, constructor, _builder.toString());
      } else {
        final Consumer<Operation> _function_2 = (Operation it) -> {
          Behavior _head = IterableExtensions.<Behavior>head(it.getMethods());
          OpaqueBehavior opaque = ((OpaqueBehavior) _head);
          if ((opaque != null)) {
            String body = opaque.getBodies().get(0);
            StringConcatenation _builder_1 = new StringConcatenation();
            _builder_1.append(MonitoringTransformation.MONITOR_SOCKET_NAME);
            _builder_1.append(" = socket(AF_INET, SOCK_DGRAM, 17);");
            _builder_1.newLineIfNotEmpty();
            _builder_1.append(MonitoringTransformation.MONITOR_ADDRESS_NAME);
            _builder_1.append(".sin_addr.s_addr = inet_addr(\"");
            String _ipAddress_1 = MonitoringPreferencesUtil.getIpAddress();
            _builder_1.append(_ipAddress_1);
            _builder_1.append("\");");
            _builder_1.newLineIfNotEmpty();
            _builder_1.append(MonitoringTransformation.MONITOR_ADDRESS_NAME);
            _builder_1.append(".sin_family = AF_INET;");
            _builder_1.newLineIfNotEmpty();
            _builder_1.append(MonitoringTransformation.MONITOR_ADDRESS_NAME);
            _builder_1.append(".sin_port = htons(");
            int _port_1 = MonitoringPreferencesUtil.getPort();
            _builder_1.append(_port_1);
            _builder_1.append(");");
            _builder_1.newLineIfNotEmpty();
            _builder_1.append(body);
            this.core.createOpaqueBehavior(this.superContext, it, _builder_1.toString());
          } else {
            StringConcatenation _builder_2 = new StringConcatenation();
            _builder_2.append(MonitoringTransformation.MONITOR_SOCKET_NAME);
            _builder_2.append(" = socket(AF_INET, SOCK_DGRAM, 17);");
            _builder_2.newLineIfNotEmpty();
            _builder_2.append(MonitoringTransformation.MONITOR_ADDRESS_NAME);
            _builder_2.append(".sin_addr.s_addr = inet_addr(\"");
            String _ipAddress_2 = MonitoringPreferencesUtil.getIpAddress();
            _builder_2.append(_ipAddress_2);
            _builder_2.append("\");");
            _builder_2.newLineIfNotEmpty();
            _builder_2.append(MonitoringTransformation.MONITOR_ADDRESS_NAME);
            _builder_2.append(".sin_family = AF_INET;");
            _builder_2.newLineIfNotEmpty();
            _builder_2.append(MonitoringTransformation.MONITOR_ADDRESS_NAME);
            _builder_2.append(".sin_port = htons(");
            int _port_2 = MonitoringPreferencesUtil.getPort();
            _builder_2.append(_port_2);
            _builder_2.append(");");
            this.core.createOpaqueBehavior(this.superContext, it, _builder_2.toString());
          }
        };
        targetContructors.forEach(_function_2);
      }
    }
  }

  public String generateTransitionCode(final Transition t) {
    if (this.isMonitored) {
      boolean monitorThis = true;
      final StateMachine stateMachine = t.containingStateMachine();
      if ((stateMachine != null)) {
        final Monitored monitoredStereotype = UMLUtil.<Monitored>getStereotypeApplication(stateMachine, Monitored.class);
        if ((monitoredStereotype != null)) {
          monitorThis = monitoredStereotype.isGenerateMonitoringCode();
          if (monitorThis) {
            boolean _contains = monitoredStereotype.getGenerateExclude().contains(t);
            boolean _not = (!_contains);
            monitorThis = _not;
            if ((monitorThis && (!monitoredStereotype.getGenerateExclusivelyInclude().isEmpty()))) {
              monitorThis = monitoredStereotype.getGenerateExclusivelyInclude().contains(t);
            }
          }
        }
      }
      if (monitorThis) {
        if (((t.getQualifiedName() != null) && (!t.getQualifiedName().isEmpty()))) {
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("if (");
          _builder.append(MonitoringTransformation.MONITOR_SOCKET_NAME);
          _builder.append(" != -1) {");
          _builder.newLineIfNotEmpty();
          _builder.append("\t");
          _builder.append("sendto(");
          _builder.append(MonitoringTransformation.MONITOR_SOCKET_NAME, "\t");
          _builder.append(", \"TRANSITION|");
          String _qualifiedName = t.getQualifiedName();
          _builder.append(_qualifiedName, "\t");
          _builder.append("\", sizeof(\"TRANSITION|");
          String _qualifiedName_1 = t.getQualifiedName();
          _builder.append(_qualifiedName_1, "\t");
          _builder.append("\"), 0, reinterpret_cast<const sockaddr*>(&");
          _builder.append(MonitoringTransformation.MONITOR_ADDRESS_NAME, "\t");
          _builder.append("), sizeof(");
          _builder.append(MonitoringTransformation.MONITOR_ADDRESS_NAME, "\t");
          _builder.append("));");
          _builder.newLineIfNotEmpty();
          _builder.append("}");
          _builder.newLine();
          return _builder.toString();
        }
      }
    }
    return "";
  }
}
