/**
 * Copyright (c) 2022, 2023 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Mohamed Harkat - Initial API and implementation
 *   Ansgar Radermacher - Integration and bug fixes
 */
package org.eclipse.papyrus.designer.languages.python.codegen.gen;

import java.util.Objects;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.OpaqueBehavior;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.profile.standard.Create;
import org.eclipse.uml2.uml.profile.standard.Destroy;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class PyMethods {
  public static CharSequence method(final Operation method) {
    boolean _isStatic = method.isStatic();
    if (_isStatic) {
      return PyMethods.staticMethod(method);
    } else {
      return PyMethods.normalMethod(method);
    }
  }

  /**
   * Write the header of a static method
   */
  public static CharSequence staticMethod(final Operation method) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("@staticmethod");
    _builder.newLine();
    _builder.append("def ");
    String _writeMethodVisibility = PyMethods.writeMethodVisibility(method);
    _builder.append(_writeMethodVisibility);
    String _name = method.getName();
    _builder.append(_name);
    _builder.append("(");
    {
      Iterable<Parameter> _nonRetParams = PyMethods.nonRetParams(method);
      boolean _hasElements = false;
      for(final Parameter parameter : _nonRetParams) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(", ", "");
        }
        String _name_1 = parameter.getName();
        _builder.append(_name_1);
      }
    }
    _builder.append("):");
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  /**
   * Write the header of a normal method, the self keyword should be added as a parameter
   */
  public static CharSequence normalMethod(final Operation method) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("def ");
    String _writeMethodVisibility = PyMethods.writeMethodVisibility(method);
    _builder.append(_writeMethodVisibility);
    String _mName = PyMethods.mName(method);
    _builder.append(_mName);
    _builder.append("(self");
    {
      Iterable<Parameter> _nonRetParams = PyMethods.nonRetParams(method);
      for(final Parameter parameter : _nonRetParams) {
        _builder.append(", ");
        String _name = parameter.getName();
        _builder.append(_name);
      }
    }
    _builder.append("):");
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  /**
   * Write the method's name. if the method is a constructor or a destroyer, then replace the name with "__init__" or
   * "__del__" respectively. Else, write the method's name
   */
  public static String mName(final Operation method) {
    String _xifexpression = null;
    boolean _isApplied = StereotypeUtil.isApplied(method, Create.class);
    if (_isApplied) {
      _xifexpression = "__init__";
    } else {
      String _xifexpression_1 = null;
      boolean _isApplied_1 = StereotypeUtil.isApplied(method, Destroy.class);
      if (_isApplied_1) {
        _xifexpression_1 = "__del__";
      } else {
        _xifexpression_1 = method.getName();
      }
      _xifexpression = _xifexpression_1;
    }
    return _xifexpression;
  }

  /**
   * Write the mehtod's body. It ensures that the body is written in Python.
   */
  public static String mBody(final Operation operation) {
    EList<Behavior> _methods = operation.getMethods();
    for (final Behavior method : _methods) {
      if ((method instanceof OpaqueBehavior)) {
        for (int i = 0; (i < ((OpaqueBehavior)method).getBodies().size()); i++) {
          String _get = ((OpaqueBehavior)method).getLanguages().get(i);
          boolean _equals = Objects.equals(_get, "Python");
          if (_equals) {
            return ((OpaqueBehavior)method).getBodies().get(i);
          }
        }
      }
    }
    return null;
  }

  public static Iterable<Parameter> nonRetParams(final Operation method) {
    final Function1<Parameter, Boolean> _function = (Parameter it) -> {
      ParameterDirectionKind _direction = it.getDirection();
      return Boolean.valueOf((!Objects.equals(_direction, ParameterDirectionKind.RETURN_LITERAL)));
    };
    return IterableExtensions.<Parameter>filter(method.getOwnedParameters(), _function);
  }

  /**
   * Check if the method is a Constructor
   */
  public static boolean isConstructor(final Operation method) {
    boolean _isApplied = StereotypeUtil.isApplied(method, Create.class);
    if (_isApplied) {
      return true;
    } else {
      return false;
    }
  }

  /**
   * Check if the method is a destroyer
   */
  public static boolean isDestroyer(final Operation method) {
    boolean _isApplied = StereotypeUtil.isApplied(method, Destroy.class);
    if (_isApplied) {
      return true;
    } else {
      return false;
    }
  }

  /**
   * Write the visibility of the method.
   * __ for private
   * _ for protected
   * nothing is adding for public visibility
   */
  public static String writeMethodVisibility(final Operation method) {
    String _name = method.getVisibility().name();
    if (_name != null) {
      switch (_name) {
        case "PRIVATE_LITERAL":
          StringConcatenation _builder = new StringConcatenation();
          _builder.append("__");
          return _builder.toString();
        case "PROTECTED_LITERAL":
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("_");
          return _builder_1.toString();
        default:
          StringConcatenation _builder_2 = new StringConcatenation();
          return _builder_2.toString();
      }
    } else {
      StringConcatenation _builder_2 = new StringConcatenation();
      return _builder_2.toString();
    }
  }
}
