/**
 * Copyright (c) 2016 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Shuai Li (CEA LIST) <shuai.li@cea.fr> - Initial API and implementation
 *   Van Cam Pham (CEA LIST) <vancam.pham@cea.fr> - Reverse implementation
 */
package org.eclipse.papyrus.designer.languages.cpp.reverse.utils;

import com.google.common.collect.Iterables;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Consumer;
import org.eclipse.cdt.core.dom.ast.ExpansionOverlapsBoundaryException;
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTFunctionDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTName;
import org.eclipse.cdt.core.dom.ast.IASTParameterDeclaration;
import org.eclipse.cdt.core.model.ICElement;
import org.eclipse.cdt.core.model.IParent;
import org.eclipse.cdt.core.model.IStructure;
import org.eclipse.cdt.core.model.IStructureTemplate;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.cdt.core.parser.IToken;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.common.util.UniqueEList;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.papyrus.designer.languages.common.profile.Codegen.NoCodeGen;
import org.eclipse.papyrus.designer.languages.cpp.codegen.transformation.CppLocationStrategy;
import org.eclipse.papyrus.designer.languages.cpp.library.CppUriConstants;
import org.eclipse.papyrus.designer.languages.cpp.reverse.BatchReverseFunctionBody;
import org.eclipse.papyrus.designer.languages.cpp.reverse.ReverseUtils;
import org.eclipse.papyrus.designer.uml.tools.utils.PackageUtil;
import org.eclipse.papyrus.designer.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Model;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.PackageableElement;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.Type;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.ExclusiveRange;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Utility methods for round-trip
 */
@SuppressWarnings("all")
public class RoundtripCppUtils {
  /**
   * Get or create a classifier
   * Use for Source sync.
   */
  public static Classifier getOrCreateClassifier(final IStructure iStructure, final ITranslationUnit iTu, final String projectName, final Model model) {
    Classifier classifier = null;
    final String[] arrayNames = IterableExtensions.<String>lastOrNull(((Iterable<String>)Conversions.doWrapArray(iTu.getLocation().toString().split(projectName)))).substring(1).split("/");
    final ArrayList<String> names = new ArrayList<String>();
    final Consumer<String> _function = (String it) -> {
      names.add(it);
    };
    ((List<String>)Conversions.doWrapArray(arrayNames)).forEach(_function);
    if ((classifier == null)) {
      org.eclipse.uml2.uml.Package parentPack = ReverseUtils.getCorrespondingModel(iTu);
      Type _ownedType = parentPack.getOwnedType(iStructure.getElementName());
      boolean _tripleEquals = (_ownedType == null);
      if (_tripleEquals) {
        parentPack.createOwnedClass(iStructure.getElementName(), false);
      }
      Type _ownedType_1 = parentPack.getOwnedType(iStructure.getElementName());
      classifier = ((Classifier) _ownedType_1);
    }
    return classifier;
  }

  public static String getFileName(final NamedElement element) {
    final CppLocationStrategy cppLocationStrategy = new CppLocationStrategy();
    return cppLocationStrategy.getFileName(element);
  }

  public static Type getPrimitiveType(final String name, final Model model) {
    boolean isPrimitive = RoundtripCppUtils.isPrimitiveCppType(name);
    if (isPrimitive) {
      org.eclipse.uml2.uml.Package ansiPack = PackageUtil.loadPackage(CppUriConstants.ANSIC_LIB_URI, model.eResource().getResourceSet());
      return RoundtripCppUtils.getTypeFromModel(name, ansiPack);
    }
    return null;
  }

  public static boolean isPrimitiveCppType(final String name) {
    final Function1<String, Boolean> _function = (String it) -> {
      return Boolean.valueOf(it.equals(name));
    };
    boolean _isEmpty = IterableExtensions.isEmpty(IterableExtensions.<String>filter(((Iterable<String>)Conversions.doWrapArray(BatchReverseFunctionBody.ansiTypes)), _function));
    return (!_isEmpty);
  }

  public static Type getTypeFromModel(final String name, final org.eclipse.uml2.uml.Package pack) {
    if ((pack == null)) {
      return null;
    }
    PackageableElement ret = pack.getPackagedElement(name);
    if (((ret == null) || (!(ret instanceof Type)))) {
      return null;
    }
    return ((Type) ret);
  }

  public static EList<EObject> applyProfile(final Model model, final String uri) {
    try {
      EList<EObject> _xblockexpression = null;
      {
        Resource resource = model.eResource().getResourceSet().getResource(URI.createURI(uri), false);
        if ((resource == null)) {
          resource = model.eResource().getResourceSet().createResource(URI.createURI(uri));
          resource.load(null);
        }
        Profile profile = IterableExtensions.<Profile>head(Iterables.<Profile>filter(resource.getContents(), Profile.class));
        EList<EObject> _xifexpression = null;
        if (((profile != null) && (!model.isProfileApplied(profile)))) {
          _xifexpression = model.applyProfile(profile);
        }
        _xblockexpression = _xifexpression;
      }
      return _xblockexpression;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public static Operation findMatchOperation(final Classifier classifier, final IASTFunctionDeclarator declarator, final String[] names, final ITranslationUnit iTu) {
    try {
      Operation ret = null;
      if (((((classifier != null) && (declarator != null)) && (names != null)) && (iTu != null))) {
        int _length = names.length;
        boolean _equals = (_length == 1);
        if (_equals) {
          return null;
        } else {
          int _length_1 = names.length;
          boolean _greaterThan = (_length_1 > 1);
          if (_greaterThan) {
            int _length_2 = names.length;
            int _minus = (_length_2 - 2);
            String classifierName = names[_minus];
            boolean _equals_1 = classifier.getName().equals(classifierName);
            boolean _not = (!_equals_1);
            if (_not) {
              return null;
            }
            List<String> contextNamespaces = new UniqueEList<String>();
            String qualifiedTypeName = "";
            int _length_3 = names.length;
            int _minus_1 = (_length_3 - 1);
            ExclusiveRange _doubleDotLessThan = new ExclusiveRange(0, _minus_1, true);
            for (final Integer i : _doubleDotLessThan) {
              {
                String _qualifiedTypeName = qualifiedTypeName;
                String _get = names[(i).intValue()];
                qualifiedTypeName = (_qualifiedTypeName + _get);
                int _length_4 = names.length;
                int _minus_2 = (_length_4 - 2);
                boolean _notEquals = ((i).intValue() != _minus_2);
                if (_notEquals) {
                  String _qualifiedTypeName_1 = qualifiedTypeName;
                  qualifiedTypeName = (_qualifiedTypeName_1 + "::");
                }
              }
            }
            contextNamespaces.add(qualifiedTypeName);
            Type ownerClass = ReverseUtils.getUMLType(classifierName, iTu);
            if ((ownerClass instanceof Classifier)) {
              boolean _notEquals = (!Objects.equals(ownerClass, classifier));
              if (_notEquals) {
                return null;
              }
            }
          }
        }
        int nParmeters = IterableExtensions.size(Iterables.<IASTParameterDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(declarator.getChildren())), IASTParameterDeclaration.class));
        final Iterable<IASTParameterDeclaration> parameterFunctionList = Iterables.<IASTParameterDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(declarator.getChildren())), IASTParameterDeclaration.class);
        int _size = IterableExtensions.size(Iterables.<IASTParameterDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(declarator.getChildren())), IASTParameterDeclaration.class));
        boolean _equals_2 = (_size == 1);
        if (_equals_2) {
          IASTParameterDeclaration parameterNode = ((IASTParameterDeclaration[])Conversions.unwrapArray((Iterables.<IASTParameterDeclaration>filter(((Iterable<?>)Conversions.doWrapArray(declarator.getChildren())), IASTParameterDeclaration.class)), IASTParameterDeclaration.class))[0];
          IASTDeclSpecifier _declSpecifier = parameterNode.getDeclSpecifier();
          boolean _tripleNotEquals = (_declSpecifier != null);
          if (_tripleNotEquals) {
            String type = "";
            IToken tokens = parameterNode.getDeclSpecifier().getSyntax();
            while ((tokens != null)) {
              {
                String _type = type;
                String _string = tokens.toString();
                type = (_type + _string);
                tokens = tokens.getNext();
              }
            }
            boolean _equals_3 = type.trim().equals("void");
            if (_equals_3) {
              nParmeters = 0;
            }
          }
        }
        final int numberParameter = nParmeters;
        int _length_4 = names.length;
        int _minus_2 = (_length_4 - 1);
        final String functionName = names[_minus_2];
        final Function1<Operation, Boolean> _function = (Operation it) -> {
          return Boolean.valueOf(it.getName().equals(functionName));
        };
        Iterable<Operation> sameNames = IterableExtensions.<Operation>filter(classifier.getAllOperations(), _function);
        final Function1<Operation, Boolean> _function_1 = (Operation it) -> {
          final Function1<Parameter, Boolean> _function_2 = (Parameter it_1) -> {
            ParameterDirectionKind _direction = it_1.getDirection();
            return Boolean.valueOf((!Objects.equals(_direction, ParameterDirectionKind.RETURN_LITERAL)));
          };
          int _size_1 = IterableExtensions.size(IterableExtensions.<Parameter>filter(it.getOwnedParameters(), _function_2));
          return Boolean.valueOf((_size_1 == numberParameter));
        };
        Iterable<Operation> sameNumberOfParameters = IterableExtensions.<Operation>filter(sameNames, _function_1);
        for (final Operation op : sameNumberOfParameters) {
          if ((ret == null)) {
            boolean isMatch = true;
            final Function1<Parameter, Boolean> _function_2 = (Parameter it) -> {
              ParameterDirectionKind _direction = it.getDirection();
              return Boolean.valueOf((!Objects.equals(_direction, ParameterDirectionKind.RETURN_LITERAL)));
            };
            Iterable<Parameter> parameterList = IterableExtensions.<Parameter>filter(op.getOwnedParameters(), _function_2);
            for (int i_1 = 0; (i_1 < numberParameter); i_1++) {
              final Iterable<Parameter> _converted_parameterList = (Iterable<Parameter>)parameterList;
              boolean _equals_4 = (((Parameter[])Conversions.unwrapArray(_converted_parameterList, Parameter.class))[i_1]).getType().getName().equals(
                RoundtripCppUtils.getCppTypeName((((IASTParameterDeclaration[])Conversions.unwrapArray(parameterFunctionList, IASTParameterDeclaration.class))[i_1]).getDeclSpecifier()));
              boolean _not_1 = (!_equals_4);
              if (_not_1) {
                isMatch = false;
              } else {
                if ((((((Parameter[])Conversions.unwrapArray(parameterList, Parameter.class))[i_1]).getName() == null) || (((Parameter[])Conversions.unwrapArray(parameterList, Parameter.class))[i_1]).getName().equals(""))) {
                  IASTName paramName = (((IASTParameterDeclaration[])Conversions.unwrapArray(parameterFunctionList, IASTParameterDeclaration.class))[i_1]).getDeclarator().getName();
                  if ((paramName != null)) {
                    final Iterable<Parameter> _converted_parameterList_1 = (Iterable<Parameter>)parameterList;
                    Parameter _get = ((Parameter[])Conversions.unwrapArray(_converted_parameterList_1, Parameter.class))[i_1];
                    _get.setName(paramName.toString());
                  }
                }
              }
            }
            if (isMatch) {
              ret = op;
            }
          }
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  private static String getCppTypeName(final IASTDeclSpecifier declarator) {
    String parameterTypeName = "";
    try {
      IToken token = declarator.getSyntax();
      while ((token != null)) {
        {
          String tokenStr = token.toString();
          boolean _equals = tokenStr.equals("*");
          if (_equals) {
          } else {
            boolean _equals_1 = tokenStr.equals("&");
            if (_equals_1) {
            } else {
              boolean _equals_2 = tokenStr.equals("const");
              if (_equals_2) {
              } else {
                boolean _equals_3 = tokenStr.equals("volatile");
                if (_equals_3) {
                } else {
                  int _length = parameterTypeName.length();
                  boolean _greaterThan = (_length > 0);
                  if (_greaterThan) {
                    String _parameterTypeName = parameterTypeName;
                    parameterTypeName = (_parameterTypeName + " ");
                  }
                  String _parameterTypeName_1 = parameterTypeName;
                  parameterTypeName = (_parameterTypeName_1 + tokenStr);
                }
              }
            }
          }
          token = token.getNext();
        }
      }
    } catch (final Throwable _t) {
      if (_t instanceof ExpansionOverlapsBoundaryException) {
      } else {
        throw Exceptions.sneakyThrow(_t);
      }
    }
    return parameterTypeName;
  }

  public static List<IStructureTemplate> getAllStructureTemplates(final IParent parent) {
    try {
      List<IStructureTemplate> ret = new UniqueEList<IStructureTemplate>();
      ICElement[] childrend = parent.getChildren();
      for (int i = 0; (i < childrend.length); i++) {
        {
          ICElement child = childrend[i];
          boolean _matched = false;
          if ((child instanceof IStructureTemplate)) {
            _matched=true;
            ICElement _get = childrend[i];
            IStructureTemplate istructureTemplate = ((IStructureTemplate) _get);
            ret.add(istructureTemplate);
          }
          if (!_matched) {
            if ((child instanceof IParent)) {
              _matched=true;
              ret.addAll(RoundtripCppUtils.getAllStructureTemplates(((IParent) child)));
            }
          }
        }
      }
      return ret;
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  public static final String EXTERNAL_PACKAGE_NAME = "external";

  public static org.eclipse.uml2.uml.Package getOrcreateExternalPackage(final org.eclipse.uml2.uml.Package parentPack, final boolean create) {
    org.eclipse.uml2.uml.Package _xblockexpression = null;
    {
      org.eclipse.uml2.uml.Package _nestedPackage = parentPack.getNestedPackage(RoundtripCppUtils.EXTERNAL_PACKAGE_NAME);
      boolean _tripleEquals = (_nestedPackage == null);
      if (_tripleEquals) {
        if (create) {
          org.eclipse.uml2.uml.Package createdPack = parentPack.createNestedPackage(RoundtripCppUtils.EXTERNAL_PACKAGE_NAME);
          StereotypeUtil.apply(createdPack, NoCodeGen.class);
        }
      }
      _xblockexpression = parentPack.getNestedPackage(RoundtripCppUtils.EXTERNAL_PACKAGE_NAME);
    }
    return _xblockexpression;
  }
}
