/**
 * Copyright (c) 2020, 2023, 2024 CEA LIST.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Matteo MORELLI (CEA LIST) <matteo.morelli@cea.fr> - Bug #566899
 *  Matteo MORELLI (CEA LIST) <matteo.morelli@cea.fr> - Bug #581690
 *  Matteo MORELLI (CEA LIST) <matteo.morelli@cea.fr> - #12/Simplify the process to execute a system task
 */
package org.eclipse.papyrus.robotics.ros2.codegen.common.utils;

import com.google.common.collect.Iterables;
import java.util.Arrays;
import java.util.LinkedHashSet;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.common.util.URI;
import org.eclipse.papyrus.robotics.codegen.common.utils.PackageTools;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.profile.robotics.behavior.Task;
import org.eclipse.papyrus.robotics.profile.robotics.services.CoordinationService;
import org.eclipse.papyrus.robotics.profile.robotics.skills.InAttribute;
import org.eclipse.papyrus.robotics.profile.robotics.skills.OutAttribute;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillDefinition;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillOperationalState;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillParameter;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillSemantic;
import org.eclipse.uml2.uml.DataType;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.Namespace;
import org.eclipse.uml2.uml.PrimitiveType;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class SkillUtils {
  /**
   * Determine the file name of C++ code that realizes the skill definition by means of its default semantics
   */
  public static String realizationPackageName(final org.eclipse.uml2.uml.Package pkg) {
    try {
      if ((pkg == null)) {
        throw new Exception("Model error! Can\'\'t access the skill realization model!");
      }
      String _pkgName = PackageTools.pkgName(pkg);
      return (_pkgName + "_skillrealizations");
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * Return the unique skills that the system needs to execute
   */
  public static LinkedHashSet<SkillDefinition> getUniqueSkills(final org.eclipse.papyrus.robotics.profile.robotics.components.System sys) {
    LinkedHashSet<SkillDefinition> res = new LinkedHashSet<SkillDefinition>();
    if ((sys != null)) {
      EList<Task> _task = sys.getTask();
      for (final Task task : _task) {
        res.addAll(task.getSkills());
      }
    }
    return res;
  }

  /**
   * Get the skill definition name from the base UML::Operation
   */
  public static String getName(final SkillDefinition sdef) {
    if ((sdef != null)) {
      return sdef.getBase_Operation().getName();
    }
    return "";
  }

  /**
   * Return the type of a skill definition parameter as a UML::DataType
   */
  public static DataType getType(final SkillParameter param) {
    if ((param != null)) {
      Type _type = param.getBase_Parameter().getType();
      if ((_type instanceof DataType)) {
        Type _type_1 = param.getBase_Parameter().getType();
        return ((DataType) _type_1);
      }
    }
    return null;
  }

  /**
   * Return the list of unique skill parameter types for a given skill definition
   */
  public static LinkedHashSet<DataType> getUniqueSkillParameterTypes(final SkillDefinition skill) {
    LinkedHashSet<DataType> res = new LinkedHashSet<DataType>();
    if ((skill != null)) {
      EList<InAttribute> _ins = skill.getIns();
      EList<OutAttribute> _outs = skill.getOuts();
      Iterable<SkillParameter> _plus = Iterables.<SkillParameter>concat(_ins, _outs);
      for (final SkillParameter p : _plus) {
        {
          final DataType paramType = SkillUtils.getType(p);
          if ((!(paramType instanceof PrimitiveType))) {
            res.add(paramType);
          }
        }
      }
    }
    return res;
  }

  /**
   * Return the ROS 2 header file name from a data type (communication object).
   */
  public static String getROS2TypeIncludePath(final DataType tp) {
    if ((tp != null)) {
      String _escapeCamlCase = RosHelpers.escapeCamlCase(MessageUtils.getROS2qMsgName(tp));
      return (_escapeCamlCase + ".hpp");
    }
    return "";
  }

  /**
   * Return the ROS2 header file name from a data type (communication object).
   */
  public static String getROS2TypeFromMsgName(final DataType tp) {
    if ((tp != null)) {
      return MessageUtils.getROS2qMsgName(tp).replace("/", Namespace.SEPARATOR);
    }
    return "";
  }

  /**
   * Return the ROS 2 package that contains the .behaviortreeschema model of first item in the System's task list
   * the path to the concrete .behaviortreeschema model in the package
   */
  public static String getFilePathCoordinatesOfDefaultTask(final org.eclipse.papyrus.robotics.profile.robotics.components.System sys) {
    String res = "";
    if ((sys != null)) {
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(sys.getTask());
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        final Task default_task = sys.getTask().get(0);
        final URI uri = default_task.getBase_Class().eResource().getURI();
        res = uri.segment(1);
      }
    }
    return res;
  }

  /**
   * Return the path to the folder that contains the concrete .behaviortreeschema model of first item in the System's task list
   * 
   * The path is either the full path relative to its ROS 2 package root, or just the name of the folder that contains the .behaviortreeschema model
   */
  public static String getPackageRelativePathOfDefaultTask(final org.eclipse.papyrus.robotics.profile.robotics.components.System sys, final boolean fullpath) {
    String res = "";
    if ((sys != null)) {
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(sys.getTask());
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        final Task default_task = sys.getTask().get(0);
        final URI uri = default_task.getBase_Class().eResource().getURI();
        if ((!fullpath)) {
          int _segmentCount = uri.segmentCount();
          int _minus = (_segmentCount - 2);
          res = uri.segment(_minus);
        } else {
          Path path = new Path("");
          String[] _segments = uri.segments();
          int _segmentCount_1 = uri.segmentCount();
          int _minus_1 = (_segmentCount_1 - 1);
          final String[] pertinentSegments = Arrays.<String>copyOfRange(_segments, 2, _minus_1);
          for (final String s : pertinentSegments) {
            IPath _append = path.append(s);
            path = ((Path) _append);
          }
          res = path.toOSString();
        }
      }
    }
    return res;
  }

  /**
   * Return the filename of .behaviortreeschema model
   */
  public static String getBTFileNameOfDefaultTask(final org.eclipse.papyrus.robotics.profile.robotics.components.System sys) {
    String res = "";
    if ((sys != null)) {
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(sys.getTask());
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        final Task default_task = sys.getTask().get(0);
        String modelname = default_task.getBase_Class().getName();
        IPath _addFileExtension = new Path(modelname).addFileExtension("behaviortreeschema");
        Path path = ((Path) _addFileExtension);
        res = path.toOSString();
      }
    }
    return res;
  }

  /**
   * Return the name of a skill definition parameter
   */
  public static String getName(final SkillParameter param) {
    if ((param != null)) {
      return param.getBase_Parameter().getName();
    }
    return null;
  }

  /**
   * Return the first operational state of skill FSM
   */
  public static SkillOperationalState getFirstOpState(final SkillSemantic sem) {
    if ((sem != null)) {
      return sem.getOperational().get(0);
    }
    return null;
  }

  /**
   * Check whether a skill operational state will access underlying SW components
   * by means of a configuration and coordination interface
   */
  public static boolean doesConfigAndCoordOfComponents(final SkillOperationalState ops) {
    if ((ops != null)) {
      CoordinationService _compInterface = ops.getCompInterface();
      boolean _tripleNotEquals = (_compInterface != null);
      if (_tripleNotEquals) {
        return true;
      }
    }
    return false;
  }

  /**
   * Return the qualified name of coordination and configuration interface used by a skill operational state
   */
  public static String getCoordinationIfQn(final SkillOperationalState ops) {
    boolean _doesConfigAndCoordOfComponents = SkillUtils.doesConfigAndCoordOfComponents(ops);
    if (_doesConfigAndCoordOfComponents) {
      final String qName = MessageUtils.getROS2qMsgName(ops.getCompInterface().getBase_Interface());
      return qName.replace("/", "::");
    }
    return "";
  }

  /**
   * Return the qualified name of coordination and configuration interface used by a skill operational state
   */
  public static String getCoordinationIfIncludePath(final SkillOperationalState ops) {
    boolean _doesConfigAndCoordOfComponents = SkillUtils.doesConfigAndCoordOfComponents(ops);
    if (_doesConfigAndCoordOfComponents) {
      final String qName = MessageUtils.getROS2qMsgName(ops.getCompInterface().getBase_Interface());
      String _escapeCamlCase = RosHelpers.escapeCamlCase(qName);
      return (_escapeCamlCase + ".hpp");
    }
    return "";
  }

  /**
   * Return the qualified name of coordination and configuration interface used by a skill operational state
   */
  public static Interface getCompInterface(final SkillOperationalState ops) {
    boolean _doesConfigAndCoordOfComponents = SkillUtils.doesConfigAndCoordOfComponents(ops);
    if (_doesConfigAndCoordOfComponents) {
      return ops.getCompInterface().getBase_Interface();
    }
    return null;
  }

  /**
   * Determine the file name of C++ code that realizes the skill definition by means of its default semantics
   */
  public static String realizationFileName(final SkillDefinition sdef) {
    try {
      if ((sdef == null)) {
        throw new Exception("Model error! Found UML::Operation with no stereotype applied!");
      }
      final SkillOperationalState ops = SkillUtils.getFirstOpState(sdef.getDefaultSemantic());
      String _xifexpression = null;
      if ((SkillUtils.doesConfigAndCoordOfComponents(ops) && 
        InteractionUtils.isQuery(InteractionUtils.getCommunicationPattern(SkillUtils.getCompInterface(ops))))) {
        _xifexpression = "_condition";
      } else {
        _xifexpression = "_action";
      }
      final String suffix = _xifexpression;
      String _lowerCase = String.join("_", sdef.getBase_Operation().getName().split("(?<=.)(?=\\p{Lu})")).toLowerCase();
      return (_lowerCase + suffix);
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }

  /**
   * doesConfigAndCoordOfComponents() on UML::State
   */
  public static boolean doesConfigAndCoordOfComponents(final State st) {
    return SkillUtils.doesConfigAndCoordOfComponents(UMLUtil.<SkillOperationalState>getStereotypeApplication(st, SkillOperationalState.class));
  }

  /**
   * getUniqueSkills() on UML::Class
   */
  public static LinkedHashSet<SkillDefinition> getUniqueSkills(final org.eclipse.uml2.uml.Class sys) {
    if ((sys == null)) {
      return null;
    }
    return SkillUtils.getUniqueSkills(UMLUtil.<org.eclipse.papyrus.robotics.profile.robotics.components.System>getStereotypeApplication(sys, org.eclipse.papyrus.robotics.profile.robotics.components.System.class));
  }

  public static String getROS2PackageNameOfDefaultTask(final org.eclipse.uml2.uml.Class sys) {
    if ((sys == null)) {
      return null;
    }
    return SkillUtils.getFilePathCoordinatesOfDefaultTask(UMLUtil.<org.eclipse.papyrus.robotics.profile.robotics.components.System>getStereotypeApplication(sys, org.eclipse.papyrus.robotics.profile.robotics.components.System.class));
  }

  public static String getPackageRelativePathOfDefaultTask(final org.eclipse.uml2.uml.Class sys) {
    if ((sys == null)) {
      return null;
    }
    return SkillUtils.getPackageRelativePathOfDefaultTask(UMLUtil.<org.eclipse.papyrus.robotics.profile.robotics.components.System>getStereotypeApplication(sys, org.eclipse.papyrus.robotics.profile.robotics.components.System.class), true);
  }

  public static String getPackageRelativeFolderNameOfDefaultTask(final org.eclipse.uml2.uml.Class sys) {
    if ((sys == null)) {
      return null;
    }
    return SkillUtils.getPackageRelativePathOfDefaultTask(UMLUtil.<org.eclipse.papyrus.robotics.profile.robotics.components.System>getStereotypeApplication(sys, org.eclipse.papyrus.robotics.profile.robotics.components.System.class), false);
  }

  public static String getBTFileNameOfDefaultTask(final org.eclipse.uml2.uml.Class sys) {
    if ((sys == null)) {
      return null;
    }
    return SkillUtils.getBTFileNameOfDefaultTask(UMLUtil.<org.eclipse.papyrus.robotics.profile.robotics.components.System>getStereotypeApplication(sys, org.eclipse.papyrus.robotics.profile.robotics.components.System.class));
  }
}
