/**
 * Copyright (c) 2018-2020, 2023, 2024 CEA LIST.
 * 
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 *  Matteo MORELLI      matteo.morelli@cea.fr - Bug #566899
 *  Matteo MORELLI      matteo.morelli@cea.fr - Bug #581433
 *  Matteo MORELLI      matteo.morelli@cea.fr - #12/Simplify the process to execute a system task
 */
package org.eclipse.papyrus.robotics.ros2.codegen.common.launch;

import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Objects;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.deployment.tools.DepUtils;
import org.eclipse.papyrus.designer.languages.common.base.file.IPFileSystemAccess;
import org.eclipse.papyrus.designer.languages.common.base.file.ProtSection;
import org.eclipse.papyrus.robotics.bpc.profile.bpc.Entity;
import org.eclipse.papyrus.robotics.codegen.common.utils.ComponentUtils;
import org.eclipse.papyrus.robotics.codegen.common.utils.PackageTools;
import org.eclipse.papyrus.robotics.codegen.common.utils.TopicUtils;
import org.eclipse.papyrus.robotics.core.utils.InstanceUtils;
import org.eclipse.papyrus.robotics.core.utils.InteractionUtils;
import org.eclipse.papyrus.robotics.core.utils.ParameterUtils;
import org.eclipse.papyrus.robotics.core.utils.PortUtils;
import org.eclipse.papyrus.robotics.profile.robotics.skills.SkillDefinition;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SequencerUtils;
import org.eclipse.papyrus.robotics.ros2.codegen.common.utils.SkillUtils;
import org.eclipse.papyrus.robotics.ros2.preferences.Ros2Distributions;
import org.eclipse.papyrus.uml.tools.utils.ConnectorUtil;
import org.eclipse.uml2.uml.Connector;
import org.eclipse.uml2.uml.ConnectorEnd;
import org.eclipse.uml2.uml.InstanceSpecification;
import org.eclipse.uml2.uml.InstanceValue;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Slot;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

/**
 * Create a set of launch scripts
 */
@SuppressWarnings("all")
public class LaunchScript {
  /**
   * create a launch script for a single component instance (part)
   */
  public static CharSequence createLaunchScriptSingle(final org.eclipse.uml2.uml.Class system, final Property part, final boolean addSequencer) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("from launch import LaunchDescription");
    _builder.newLine();
    {
      if (addSequencer) {
        _builder.append("from launch_ros.descriptions import ParameterFile");
      }
    }
    _builder.newLineIfNotEmpty();
    _builder.append("from launch_ros.actions import ");
    {
      if ((addSequencer || InstanceUtils.isLifecycle(part))) {
        _builder.append("Lifecycle");
      }
    }
    _builder.append("Node");
    _builder.newLineIfNotEmpty();
    _builder.append("from ament_index_python.packages import get_package_share_directory");
    _builder.newLine();
    _builder.newLine();
    _builder.append("share_dir = get_package_share_directory(\'");
    String _pkgName = PackageTools.pkgName(system.getNearestPackage());
    _builder.append(_pkgName);
    _builder.append("\')");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("# ");
    String _protSection = ProtSection.protSection("imports");
    _builder.append(_protSection);
    _builder.newLineIfNotEmpty();
    _builder.append("# ");
    String _protSection_1 = ProtSection.protSection();
    _builder.append(_protSection_1);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("def generate_entity():");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# define empty user parameter (eventually redefined in the protected section afterwards)");
    _builder.newLine();
    _builder.append("\t");
    {
      if ((!addSequencer)) {
        String _name = part.getName();
        _builder.append(_name, "\t");
      } else {
        String _sequencerName = SequencerUtils.getSequencerName(system);
        _builder.append(_sequencerName, "\t");
      }
    }
    _builder.append("_custom_params = {}");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# ");
    String _protSection_2 = ProtSection.protSection("parameters");
    _builder.append(_protSection_2, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("# ");
    String _protSection_3 = ProtSection.protSection();
    _builder.append(_protSection_3, "\t");
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# Add the actions to the launch description.");
    _builder.newLine();
    {
      if ((!addSequencer)) {
        _builder.append("\t");
        Type _type = part.getType();
        final org.eclipse.uml2.uml.Class component = ((org.eclipse.uml2.uml.Class) _type);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("return ");
        {
          boolean _isLifecycle = InstanceUtils.isLifecycle(part);
          if (_isLifecycle) {
            _builder.append("Lifecycle");
          }
        }
        _builder.append("Node(");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        {
          Ros2Distributions.RDLiteral _rosDistro = Ros2Distributions.getRosDistro();
          boolean _equals = Objects.equals(_rosDistro, Ros2Distributions.RDLiteral.ELOQUENT);
          if (_equals) {
            _builder.append("node_");
          }
        }
        _builder.append("name=\'");
        String _name_1 = part.getName();
        _builder.append(_name_1, "\t\t");
        _builder.append("\',");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("package=\'");
        String _pkgName_1 = PackageTools.pkgName(component.getNearestPackage());
        _builder.append(_pkgName_1, "\t\t");
        _builder.append("\', ");
        {
          Ros2Distributions.RDLiteral _rosDistro_1 = Ros2Distributions.getRosDistro();
          boolean _equals_1 = Objects.equals(_rosDistro_1, Ros2Distributions.RDLiteral.ELOQUENT);
          if (_equals_1) {
            _builder.append("node_");
          }
        }
        _builder.append("executable=\'");
        String _name_2 = component.getName();
        _builder.append(_name_2, "\t\t");
        {
          boolean _isRegistered = ComponentUtils.isRegistered(component);
          if (_isRegistered) {
            _builder.append("_main");
          }
        }
        _builder.append("\',");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("namespace=\'\',");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("remappings=[");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t\t");
        {
          ArrayList<String> _remaps = LaunchScript.remaps(system, part);
          boolean _hasElements = false;
          for(final String remap : _remaps) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(", ", "\t\t\t");
            }
            _builder.append(remap, "\t\t\t");
          }
        }
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("],");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("parameters=[share_dir+\'/launch/cfg/param.yaml\', ");
        String _name_3 = part.getName();
        _builder.append(_name_3, "\t\t");
        _builder.append("_custom_params],");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("output=\'screen\',");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("emulate_tty=True\t# assure that RCLCPP output gets flushed");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(")");
        _builder.newLine();
      } else {
        _builder.append("\t");
        _builder.append("return LifecycleNode(");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        {
          Ros2Distributions.RDLiteral _rosDistro_2 = Ros2Distributions.getRosDistro();
          boolean _equals_2 = Objects.equals(_rosDistro_2, Ros2Distributions.RDLiteral.ELOQUENT);
          if (_equals_2) {
            _builder.append("node_");
          }
        }
        _builder.append("name=\'");
        String _sequencerName_1 = SequencerUtils.getSequencerName(system);
        _builder.append(_sequencerName_1, "\t\t");
        _builder.append("\',");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("package=\'bt_sequencer\', ");
        {
          Ros2Distributions.RDLiteral _rosDistro_3 = Ros2Distributions.getRosDistro();
          boolean _equals_3 = Objects.equals(_rosDistro_3, Ros2Distributions.RDLiteral.ELOQUENT);
          if (_equals_3) {
            _builder.append("node_");
          }
        }
        _builder.append("executable=\'bt_sequencer\',");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("namespace=\'\',");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("parameters=[ParameterFile(share_dir+\'/launch/cfg/param.yaml\', allow_substs=True), ");
        String _sequencerName_2 = SequencerUtils.getSequencerName(system);
        _builder.append(_sequencerName_2, "\t\t");
        _builder.append("_custom_params],");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("output=\'screen\',");
        _builder.newLine();
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("emulate_tty=True\t# assure that RCLCPP output gets flushed");
        _builder.newLine();
        _builder.append("\t");
        _builder.append(")");
        _builder.newLine();
      }
    }
    _builder.newLine();
    _builder.append("def generate_launch_description():");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# Launch Description");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("ld = LaunchDescription()");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("ld.add_entity(generate_entity())");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# ");
    String _protSection_4 = ProtSection.protSection("post-launch");
    _builder.append(_protSection_4, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("# ");
    String _protSection_5 = ProtSection.protSection();
    _builder.append(_protSection_5, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("return ld");
    _builder.newLine();
    return _builder;
  }

  /**
   * Create a launch script based on the launch script for individual instances. This script is used for
   * individual activation scripts as well as for launh and activation of the whole system
   */
  public static CharSequence createLaunchScript(final org.eclipse.uml2.uml.Class system, final List<Property> parts, final boolean addSequencer, final boolean activate) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("import launch.actions");
    _builder.newLine();
    _builder.append("import launch_ros");
    _builder.newLine();
    _builder.append("import lifecycle_msgs.msg");
    _builder.newLine();
    _builder.newLine();
    _builder.append("import sys");
    _builder.newLine();
    _builder.append("import os.path");
    _builder.newLine();
    _builder.newLine();
    _builder.append("from launch_ros.events.lifecycle import ChangeState");
    _builder.newLine();
    _builder.append("from launch import LaunchDescription");
    _builder.newLine();
    _builder.append("from ament_index_python.packages import get_package_share_directory");
    _builder.newLine();
    _builder.newLine();
    _builder.append("share_dir = get_package_share_directory(\'");
    String _pkgName = PackageTools.pkgName(system.getNearestPackage());
    _builder.append(_pkgName);
    _builder.append("\')");
    _builder.newLineIfNotEmpty();
    _builder.append("# generated scripts are in launch folder");
    _builder.newLine();
    _builder.append("sys.path.append(os.path.join(share_dir, \'launch\'))");
    _builder.newLine();
    _builder.append("# load launch scripts for each instance");
    _builder.newLine();
    {
      for(final Property part : parts) {
        _builder.append("import launch_");
        String _name = part.getName();
        _builder.append(_name);
        _builder.newLineIfNotEmpty();
      }
    }
    {
      if (addSequencer) {
        _builder.append("import launch_");
        String _sequencerName = SequencerUtils.getSequencerName(system);
        _builder.append(_sequencerName);
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    _builder.append("# ");
    String _protSection = ProtSection.protSection("imports");
    _builder.append(_protSection);
    _builder.newLineIfNotEmpty();
    _builder.append("# ");
    String _protSection_1 = ProtSection.protSection();
    _builder.append(_protSection_1);
    _builder.newLineIfNotEmpty();
    _builder.newLine();
    _builder.append("def generate_launch_description():");
    _builder.newLine();
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# Launch Description");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("ld = launch.LaunchDescription()");
    _builder.newLine();
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# call entity creation for each instance");
    _builder.newLine();
    {
      for(final Property part_1 : parts) {
        _builder.append("\t");
        String _name_1 = part_1.getName();
        _builder.append(_name_1, "\t");
        _builder.append("_node = launch_");
        String _name_2 = part_1.getName();
        _builder.append(_name_2, "\t");
        _builder.append(".generate_entity()");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      if (addSequencer) {
        _builder.append("\t");
        String _sequencerName_1 = SequencerUtils.getSequencerName(system);
        _builder.append(_sequencerName_1, "\t");
        _builder.append("_node = launch_");
        String _sequencerName_2 = SequencerUtils.getSequencerName(system);
        _builder.append(_sequencerName_2, "\t");
        _builder.append(".generate_entity()");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.append("\t");
    _builder.append("# now add all obtained entities to the launch description");
    _builder.newLine();
    {
      for(final Property part_2 : parts) {
        _builder.append("\t");
        _builder.append("ld.add_entity(");
        String _name_3 = part_2.getName();
        _builder.append(_name_3, "\t");
        _builder.append("_node)");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      if (addSequencer) {
        _builder.append("\t");
        _builder.append("ld.add_entity(");
        String _sequencerName_3 = SequencerUtils.getSequencerName(system);
        _builder.append(_sequencerName_3, "\t");
        _builder.append("_node)");
        _builder.newLineIfNotEmpty();
      }
    }
    _builder.newLine();
    {
      if (activate) {
        _builder.append("\t");
        _builder.append("# transition to configure after startup");
        _builder.newLine();
        {
          for(final Property part_3 : parts) {
            {
              boolean _isLifecycle = InstanceUtils.isLifecycle(part_3);
              if (_isLifecycle) {
                _builder.append("\t");
                CharSequence _generateConfig = LaunchScript.generateConfig(part_3.getName());
                _builder.append(_generateConfig, "\t");
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
        {
          if (addSequencer) {
            _builder.append("\t");
            CharSequence _generateConfig_1 = LaunchScript.generateConfig(SequencerUtils.getSequencerName(system));
            _builder.append(_generateConfig_1, "\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.newLine();
        _builder.append("\t");
        _builder.append("# transition to activate, once inactive");
        _builder.newLine();
        {
          for(final Property part_4 : parts) {
            {
              boolean _isLifecycle_1 = InstanceUtils.isLifecycle(part_4);
              if (_isLifecycle_1) {
                _builder.append("\t");
                CharSequence _generateActivation = LaunchScript.generateActivation(part_4.getName());
                _builder.append(_generateActivation, "\t");
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
        {
          if (addSequencer) {
            _builder.append("\t");
            CharSequence _generateActivation_1 = LaunchScript.generateActivation(SequencerUtils.getSequencerName(system));
            _builder.append(_generateActivation_1, "\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.newLine();
      }
    }
    _builder.append("\t");
    _builder.newLine();
    _builder.append("\t");
    _builder.append("# ");
    String _protSection_2 = ProtSection.protSection("post-launch");
    _builder.append(_protSection_2, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("# ");
    String _protSection_3 = ProtSection.protSection();
    _builder.append(_protSection_3, "\t");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("return ld");
    _builder.newLine();
    return _builder;
  }

  /**
   * Generate the commands for node configuration given node name
   */
  public static CharSequence generateConfig(final String name) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("configure_");
    _builder.append(name);
    _builder.append(" = launch.actions.RegisterEventHandler(");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("launch.event_handlers.on_process_start.OnProcessStart(");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("target_action=");
    _builder.append(name, "\t\t");
    _builder.append("_node,");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("on_start=[");
    _builder.newLine();
    _builder.append(" \t\t\t");
    _builder.append("launch.actions.EmitEvent(");
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append("event=launch_ros.events.lifecycle.ChangeState(");
    _builder.newLine();
    _builder.append("\t\t\t\t\t");
    _builder.append("lifecycle_node_matcher=launch.events.matches_action(");
    _builder.append(name, "\t\t\t\t\t");
    _builder.append("_node),");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t\t\t\t");
    _builder.append("transition_id=lifecycle_msgs.msg.Transition.TRANSITION_CONFIGURE");
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append(")");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append(")");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("]");
    _builder.newLine();
    _builder.append("\t");
    _builder.append(")");
    _builder.newLine();
    _builder.append(")");
    _builder.newLine();
    _builder.append("ld.add_entity(configure_");
    _builder.append(name);
    _builder.append(")");
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  /**
   * Generate the commands for node activation given node name
   */
  public static CharSequence generateActivation(final String name) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("activate_");
    _builder.append(name);
    _builder.append(" = launch.actions.RegisterEventHandler(");
    _builder.newLineIfNotEmpty();
    _builder.append("\t");
    _builder.append("launch_ros.event_handlers.OnStateTransition(");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("target_lifecycle_node=");
    _builder.append(name, "\t\t");
    _builder.append("_node,");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t");
    _builder.append("start_state=\'configuring\', goal_state=\'inactive\',");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("entities=[");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append("launch.actions.EmitEvent(");
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append("event=launch_ros.events.lifecycle.ChangeState(");
    _builder.newLine();
    _builder.append("\t\t\t\t\t");
    _builder.append("lifecycle_node_matcher=launch.events.matches_action(");
    _builder.append(name, "\t\t\t\t\t");
    _builder.append("_node),");
    _builder.newLineIfNotEmpty();
    _builder.append("\t\t\t\t\t");
    _builder.append("transition_id=lifecycle_msgs.msg.Transition.TRANSITION_ACTIVATE");
    _builder.newLine();
    _builder.append("\t\t\t\t");
    _builder.append(")");
    _builder.newLine();
    _builder.append("\t\t\t");
    _builder.append(")");
    _builder.newLine();
    _builder.append("\t\t");
    _builder.append("]");
    _builder.newLine();
    _builder.append("\t");
    _builder.append(")");
    _builder.newLine();
    _builder.append(")");
    _builder.newLine();
    _builder.append("ld.add_entity(activate_");
    _builder.append(name);
    _builder.append(")");
    _builder.newLineIfNotEmpty();
    return _builder;
  }

  /**
   * initialize parameter values from instance specifications and the (implicit) system sequencer model
   * (In the current implementation, the default skill semantics realization is always selected)
   */
  public static CharSequence createParameterFile(final org.eclipse.uml2.uml.Class system) {
    StringConcatenation _builder = new StringConcatenation();
    {
      List<Property> _compInstanceList = InstanceUtils.getCompInstanceList(system);
      for(final Property part : _compInstanceList) {
        {
          if (((part.getType() instanceof org.eclipse.uml2.uml.Class) && (ParameterUtils.getAllParameters(((org.eclipse.uml2.uml.Class) part.getType())).size() > 0))) {
            String _name = part.getName();
            _builder.append(_name);
            _builder.append(":");
            _builder.newLineIfNotEmpty();
            _builder.append("  ");
            _builder.append("ros__parameters:");
            _builder.newLine();
            final ArrayList<String> names = new ArrayList<String>();
            _builder.newLineIfNotEmpty();
            {
              Type _type = part.getType();
              List<Property> _allParameters = ParameterUtils.getAllParameters(((org.eclipse.uml2.uml.Class) _type));
              for(final Property param : _allParameters) {
                String _hierarchicalName = LaunchScript.hierarchicalName(param, names);
                _builder.append(_hierarchicalName);
                _builder.append(" : ");
                String _paramValue = LaunchScript.paramValue(part, param);
                _builder.append(_paramValue);
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
      }
    }
    {
      boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(SkillUtils.getUniqueSkills(system));
      boolean _not = (!_isNullOrEmpty);
      if (_not) {
        String _sequencerName = SequencerUtils.getSequencerName(system);
        _builder.append(_sequencerName);
        _builder.append(":");
        _builder.newLineIfNotEmpty();
        _builder.append("  ");
        _builder.append("ros__parameters:");
        _builder.newLine();
        _builder.append("    ");
        _builder.append("default_bt_xml_filename: \"$(find-pkg-share ");
        String _rOS2PackageNameOfDefaultTask = SkillUtils.getROS2PackageNameOfDefaultTask(system);
        _builder.append(_rOS2PackageNameOfDefaultTask, "    ");
        _builder.append(")/");
        String _packageRelativeFolderNameOfDefaultTask = SkillUtils.getPackageRelativeFolderNameOfDefaultTask(system);
        _builder.append(_packageRelativeFolderNameOfDefaultTask, "    ");
        _builder.append("/");
        String _bTFileNameOfDefaultTask = SkillUtils.getBTFileNameOfDefaultTask(system);
        _builder.append(_bTFileNameOfDefaultTask, "    ");
        _builder.append("\"");
        _builder.newLineIfNotEmpty();
        _builder.append("    ");
        _builder.append("plugin_lib_names: [");
        _builder.newLine();
        {
          LinkedHashSet<SkillDefinition> _uniqueSkills = SkillUtils.getUniqueSkills(system);
          boolean _hasElements = false;
          for(final SkillDefinition skill : _uniqueSkills) {
            if (!_hasElements) {
              _hasElements = true;
            } else {
              _builder.appendImmediate(",", "    ");
            }
            _builder.append("    ");
            String _realizationFileName = SkillUtils.realizationFileName(skill);
            String _plus = ("  \"" + _realizationFileName);
            _builder.append(_plus, "    ");
            _builder.append("_bt_node\"");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("    ");
        _builder.append("]");
        _builder.newLine();
      }
    }
    return _builder;
  }

  /**
   * Return either the specific value configured on the instance level or the default value
   */
  public static String paramValue(final Property part, final Property parameter) {
    String value = "";
    ValueSpecification _defaultValue = part.getDefaultValue();
    if ((_defaultValue instanceof InstanceValue)) {
      ValueSpecification _defaultValue_1 = part.getDefaultValue();
      final InstanceSpecification is = ((InstanceValue) _defaultValue_1).getInstance();
      if ((is != null)) {
        EList<Slot> _slots = is.getSlots();
        for (final Slot slot : _slots) {
          if ((Objects.equals(slot.getDefiningFeature(), parameter) && (DepUtils.firstValue(slot) != null))) {
            value = DepUtils.firstValue(slot).stringValue();
          }
        }
      }
    }
    if (((value.length() == 0) && (parameter.getDefaultValue() != null))) {
      String _stringValue = parameter.getDefaultValue().stringValue();
      String _plus = (_stringValue + " # (default)");
      value = _plus;
    }
    final Entity entity = UMLUtil.<Entity>getStereotypeApplication(parameter, Entity.class);
    if (((entity != null) && (entity.getDescription() != null))) {
      boolean _contains = value.contains("#");
      boolean _not = (!_contains);
      if (_not) {
        String _value = value;
        value = (_value + " #");
      }
      String _value_1 = value;
      String _description = entity.getDescription();
      String _plus_1 = (" " + _description);
      value = (_value_1 + _plus_1);
    }
    return value;
  }

  /**
   * Calculate the YAML representation of a hierarchical name with "." separator
   * If the parent is used the first time, emit it
   */
  public static String hierarchicalName(final NamedElement ne, final List<String> names) {
    String result = "";
    String indent = "    ";
    String qName = "";
    String[] _split = ne.getName().split("\\.");
    for (final String part : _split) {
      {
        String _qName = qName;
        qName = (_qName + (part + "."));
        boolean _contains = names.contains(qName);
        boolean _not = (!_contains);
        if (_not) {
          names.add(qName);
          int _length = result.length();
          boolean _greaterThan = (_length > 4);
          if (_greaterThan) {
            String _result = result;
            result = (_result + ":\n");
          }
          String _result_1 = result;
          result = (_result_1 + (indent + part));
        }
        String _indent = indent;
        indent = (_indent + "  ");
      }
    }
    return result;
  }

  /**
   * an XML based launch script (XML)
   */
  public static CharSequence createLaunchScriptXML(final org.eclipse.uml2.uml.Class system) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("<launch>");
    _builder.newLine();
    {
      List<Property> _compInstanceList = InstanceUtils.getCompInstanceList(system);
      for(final Property part : _compInstanceList) {
        _builder.append("\t");
        Type _type = part.getType();
        final org.eclipse.uml2.uml.Class component = ((org.eclipse.uml2.uml.Class) _type);
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("<node pkg=\"");
        String _pkgName = PackageTools.pkgName(component.getNearestPackage());
        _builder.append(_pkgName, "\t");
        _builder.append("\" exec=\"");
        String _name = component.getName();
        _builder.append(_name, "\t");
        _builder.append("\" name=\"");
        String _name_1 = part.getName();
        _builder.append(_name_1, "\t");
        _builder.append("\" respawn=\"false\" output=\"screen\">");
        _builder.newLineIfNotEmpty();
        {
          ArrayList<String> _remapsXML = LaunchScript.remapsXML(system, part);
          for(final String remap : _remapsXML) {
            _builder.append("\t");
            _builder.append("\t");
            _builder.append(remap, "\t\t");
            _builder.newLineIfNotEmpty();
          }
        }
        _builder.append("\t");
        _builder.append("\t");
        _builder.append("<param from=\"$(find-pkg-share ");
        String _pkgName_1 = PackageTools.pkgName(component.getNearestPackage());
        _builder.append(_pkgName_1, "\t\t");
        _builder.append(")/launch/cfg/param.yaml\"/>");
        _builder.newLineIfNotEmpty();
        _builder.append("\t");
        _builder.append("</node>");
        _builder.newLine();
      }
    }
    _builder.append("</launch>");
    _builder.newLine();
    return _builder;
  }

  /**
   * Remappings for XML
   */
  public static ArrayList<String> remapsXML(final org.eclipse.uml2.uml.Class system, final Property part) {
    final ArrayList<String> remaps = new ArrayList<String>();
    Type _type = part.getType();
    EList<Port> _allPorts = PortUtils.getAllPorts(((org.eclipse.uml2.uml.Class) _type));
    for (final Port port : _allPorts) {
      {
        final ConnectorEnd oppositeEnd = LaunchScript.getOpposite(system, part, port);
        if ((oppositeEnd != null)) {
          final String from = TopicUtils.getTopic(port);
          final String to = TopicUtils.getTopic(part, port, oppositeEnd);
          boolean _notEquals = (!Objects.equals(from, to));
          if (_notEquals) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("<remap from=\"");
            _builder.append(from);
            _builder.append("\" to=\"");
            _builder.append(to);
            _builder.append("\"/>");
            remaps.add(_builder.toString());
          }
        }
      }
    }
    return remaps;
  }

  /**
   * Re-mappings for command line (used for debugging a binary)
   */
  public static CharSequence remapsCmd(final org.eclipse.uml2.uml.Class system, final Property part) {
    StringConcatenation _builder = new StringConcatenation();
    {
      Type _type = part.getType();
      EList<Port> _allPorts = PortUtils.getAllPorts(((org.eclipse.uml2.uml.Class) _type));
      boolean _hasElements = false;
      for(final Port port : _allPorts) {
        if (!_hasElements) {
          _hasElements = true;
        } else {
          _builder.appendImmediate(" ", "");
        }
        final ConnectorEnd oppositeEnd = LaunchScript.getOpposite(system, part, port);
        _builder.newLineIfNotEmpty();
        {
          if ((oppositeEnd != null)) {
            final String from = TopicUtils.getTopic(port);
            _builder.newLineIfNotEmpty();
            final String to = TopicUtils.getTopic(part, port, oppositeEnd);
            _builder.newLineIfNotEmpty();
            {
              boolean _notEquals = (!Objects.equals(from, to));
              if (_notEquals) {
                _builder.append("-r ");
                String _topic = TopicUtils.getTopic(port);
                _builder.append(_topic);
                _builder.append(":=");
                String _topic_1 = TopicUtils.getTopic(part, port, oppositeEnd);
                _builder.append(_topic_1);
                _builder.newLineIfNotEmpty();
              }
            }
          }
        }
      }
    }
    return _builder;
  }

  public static ArrayList<String> remaps(final org.eclipse.uml2.uml.Class system, final Property part) {
    final ArrayList<String> remaps = new ArrayList<String>();
    Type _type = part.getType();
    EList<Port> _allPorts = PortUtils.getAllPorts(((org.eclipse.uml2.uml.Class) _type));
    for (final Port port : _allPorts) {
      {
        final ConnectorEnd oppositeEnd = LaunchScript.getOpposite(system, part, port);
        if (((oppositeEnd != null) && (InteractionUtils.getCommObject(port) != null))) {
          final String from = TopicUtils.getTopic(port);
          final String to = TopicUtils.getTopic(part, port, oppositeEnd);
          boolean _equals = from.equals(to);
          boolean _not = (!_equals);
          if (_not) {
            StringConcatenation _builder = new StringConcatenation();
            _builder.append("(\'");
            _builder.append(from);
            _builder.append("\', \'");
            _builder.append(to);
            _builder.append("\')");
            remaps.add(_builder.toString());
          }
        }
      }
    }
    return remaps;
  }

  /**
   * Return opposite part, based on the assumption that there is a single port
   * targeting a port
   */
  public static ConnectorEnd getOpposite(final org.eclipse.uml2.uml.Class system, final Property part, final Port port) {
    EList<Connector> _ownedConnectors = system.getOwnedConnectors();
    for (final Connector connector : _ownedConnectors) {
      boolean _connectsPort = ConnectorUtil.connectsPort(connector, port);
      if (_connectsPort) {
        final ConnectorEnd end = ConnectorUtil.connEndNotPart(connector, part);
        if ((end != null)) {
          return end;
        }
      }
    }
    return null;
  }

  public static void generateLaunch(final IPFileSystemAccess fileAccess, final org.eclipse.uml2.uml.Class system) {
    final List<Property> parts = InstanceUtils.getCompInstanceList(system);
    boolean _isNullOrEmpty = IterableExtensions.isNullOrEmpty(SkillUtils.getUniqueSkills(system));
    final boolean addSequencer = (!_isNullOrEmpty);
    fileAccess.generateFile("launch/launch.py", LaunchScript.createLaunchScript(system, parts, addSequencer, false).toString());
    fileAccess.generateFile("launch/activate.py", LaunchScript.createLaunchScript(system, parts, addSequencer, true).toString());
    for (final Property part : parts) {
      {
        StringConcatenation _builder = new StringConcatenation();
        _builder.append("launch/launch_");
        String _name = part.getName();
        _builder.append(_name);
        _builder.append(".py");
        fileAccess.generateFile(_builder.toString(), LaunchScript.createLaunchScriptSingle(system, part, false).toString());
        boolean _isLifecycle = InstanceUtils.isLifecycle(part);
        if (_isLifecycle) {
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append("launch/activate_");
          String _name_1 = part.getName();
          _builder_1.append(_name_1);
          _builder_1.append(".py");
          fileAccess.generateFile(_builder_1.toString(), LaunchScript.createLaunchScript(system, Collections.<Property>singletonList(part), false, true).toString());
        }
      }
    }
    boolean _since = Ros2Distributions.since(Ros2Distributions.getRosDistro(), Ros2Distributions.RDLiteral.GALACTIC);
    if (_since) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("launch/");
      String _name = system.getName();
      _builder.append(_name);
      _builder.append(".launch");
      fileAccess.generateFile(_builder.toString(), LaunchScript.createLaunchScriptXML(system).toString());
    }
    if (addSequencer) {
      StringConcatenation _builder_1 = new StringConcatenation();
      _builder_1.append("launch/launch_");
      String _sequencerName = SequencerUtils.getSequencerName(system);
      _builder_1.append(_sequencerName);
      _builder_1.append(".py");
      fileAccess.generateFile(_builder_1.toString(), LaunchScript.createLaunchScriptSingle(system, null, true).toString());
    }
    fileAccess.generateFile("launch/cfg/param.yaml", LaunchScript.createParameterFile(system).toString());
  }
}
