/**
 * Copyright (c) 2014, 2015, 2023 Christian W. Damus and others.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 * Christian W. Damus - Initial API and implementation
 * Ansgar Radermacher - bug 582492, move to com.google.inject
 */
package org.eclipse.papyrus.uml.profile.assistants.generator;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Objects;
import java.util.function.Consumer;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.gmf.runtime.emf.type.core.IElementType;
import org.eclipse.papyrus.infra.filters.CompoundFilter;
import org.eclipse.papyrus.infra.filters.Filter;
import org.eclipse.papyrus.infra.filters.FiltersFactory;
import org.eclipse.papyrus.infra.filters.OperatorKind;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistantFactory;
import org.eclipse.papyrus.infra.gmfdiag.assistant.AssistedElementTypeFilter;
import org.eclipse.papyrus.infra.gmfdiag.assistant.ElementTypeFilter;
import org.eclipse.papyrus.infra.types.ElementTypeConfiguration;
import org.eclipse.papyrus.uml.filters.ProfileApplied;
import org.eclipse.papyrus.uml.filters.UMLFiltersFactory;
import org.eclipse.papyrus.uml.profile.types.generator.UML;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.xtext.xbase.lib.CollectionExtensions;
import org.eclipse.xtext.xbase.lib.CollectionLiterals;
import org.eclipse.xtext.xbase.lib.Extension;
import org.eclipse.xtext.xbase.lib.ObjectExtensions;
import org.eclipse.xtext.xbase.lib.Procedures.Procedure1;
import org.eclipse.xtext.xbase.lib.XbaseGenerated;

/**
 * Utility extensions for working with {@link Filter}s.
 */
@Singleton
@SuppressWarnings("all")
public class FiltersUtil {
  @Extension
  private static FiltersFactory filtersFactory = FiltersFactory.eINSTANCE;

  @Extension
  private static UMLFiltersFactory umlFiltersFactory = UMLFiltersFactory.eINSTANCE;

  @Extension
  private static AssistantFactory assistantFactory = AssistantFactory.eINSTANCE;

  @Inject
  @Extension
  private UML _uML;

  @Inject
  @Extension
  private ModelingAssistantProviderRule _modelingAssistantProviderRule;

  public ElementTypeFilter toElementTypeFilter(final String typeID, final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(typeID, umlProfile);
    final ElementTypeFilter _result;
    synchronized (_createCache_toElementTypeFilter) {
      if (_createCache_toElementTypeFilter.containsKey(_cacheKey)) {
        return _createCache_toElementTypeFilter.get(_cacheKey);
      }
      ElementTypeFilter _createElementTypeFilter = FiltersUtil.assistantFactory.createElementTypeFilter();
      _result = _createElementTypeFilter;
      _createCache_toElementTypeFilter.put(_cacheKey, _result);
    }
    _init_toElementTypeFilter(_result, typeID, umlProfile);
    return _result;
  }

  private final HashMap<ArrayList<?>, ElementTypeFilter> _createCache_toElementTypeFilter = CollectionLiterals.newHashMap();

  private void _init_toElementTypeFilter(final ElementTypeFilter it, final String typeID, final Profile umlProfile) {
    it.setElementTypeID(typeID);
    int _lastIndexOf = typeID.lastIndexOf(".");
    int _plus = (_lastIndexOf + 1);
    it.setName(typeID.substring(_plus));
    this._modelingAssistantProviderRule.toModelingAssistantProvider(this._uML.getRootProfile(umlProfile)).getOwnedFilters().add(it);
  }

  public ElementTypeFilter toFilter(final IElementType elementType, final Profile umlProfile) {
    return this.internalToFilter(elementType, this._uML.getRootProfile(umlProfile));
  }

  private ElementTypeFilter internalToFilter(final IElementType elementType, final Profile rootProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(elementType, rootProfile);
    final ElementTypeFilter _result;
    synchronized (_createCache_internalToFilter) {
      if (_createCache_internalToFilter.containsKey(_cacheKey)) {
        return _createCache_internalToFilter.get(_cacheKey);
      }
      ElementTypeFilter _createElementTypeFilter = FiltersUtil.assistantFactory.createElementTypeFilter();
      _result = _createElementTypeFilter;
      _createCache_internalToFilter.put(_cacheKey, _result);
    }
    _init_internalToFilter(_result, elementType, rootProfile);
    return _result;
  }

  private final HashMap<ArrayList<?>, ElementTypeFilter> _createCache_internalToFilter = CollectionLiterals.newHashMap();

  private void _init_internalToFilter(final ElementTypeFilter it, final IElementType elementType, final Profile rootProfile) {
    it.setElementTypeID(elementType.getId());
    it.setName(elementType.getDisplayName());
    this._modelingAssistantProviderRule.toModelingAssistantProvider(rootProfile).getOwnedFilters().add(it);
  }

  public ElementTypeFilter toFilter(final ElementTypeConfiguration elementType, final Profile umlProfile) {
    return this.internalToFilter(elementType, this._uML.getRootProfile(umlProfile));
  }

  private ElementTypeFilter internalToFilter(final ElementTypeConfiguration elementType, final Profile rootProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(elementType, rootProfile);
    final ElementTypeFilter _result;
    synchronized (_createCache_internalToFilter_1) {
      if (_createCache_internalToFilter_1.containsKey(_cacheKey)) {
        return _createCache_internalToFilter_1.get(_cacheKey);
      }
      ElementTypeFilter _createElementTypeFilter = FiltersUtil.assistantFactory.createElementTypeFilter();
      _result = _createElementTypeFilter;
      _createCache_internalToFilter_1.put(_cacheKey, _result);
    }
    _init_internalToFilter_1(_result, elementType, rootProfile);
    return _result;
  }

  private final HashMap<ArrayList<?>, ElementTypeFilter> _createCache_internalToFilter_1 = CollectionLiterals.newHashMap();

  private void _init_internalToFilter_1(final ElementTypeFilter it, final ElementTypeConfiguration elementType, final Profile rootProfile) {
    it.setElementTypeID(elementType.getIdentifier());
    it.setName(elementType.getName());
    this._modelingAssistantProviderRule.toModelingAssistantProvider(rootProfile).getOwnedFilters().add(it);
  }

  public CompoundFilter toFilter(final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlProfile);
    final CompoundFilter _result;
    synchronized (_createCache_toFilter) {
      if (_createCache_toFilter.containsKey(_cacheKey)) {
        return _createCache_toFilter.get(_cacheKey);
      }
      CompoundFilter _createCompoundFilter = FiltersUtil.filtersFactory.createCompoundFilter();
      _result = _createCompoundFilter;
      _createCache_toFilter.put(_cacheKey, _result);
    }
    _init_toFilter(_result, umlProfile);
    return _result;
  }

  private final HashMap<ArrayList<?>, CompoundFilter> _createCache_toFilter = CollectionLiterals.newHashMap();

  private void _init_toFilter(final CompoundFilter it, final Profile umlProfile) {
    it.setOperator(OperatorKind.OR);
    String _qualifiedName = umlProfile.getQualifiedName();
    String _plus = ("pertains to Profile " + _qualifiedName);
    it.setName(_plus);
    CollectionExtensions.<Filter>addAll(it.getOwnedFilters(), this.toAppliedFilter(umlProfile), this.toAssistedElementTypeFilter(umlProfile));
    this._modelingAssistantProviderRule.toModelingAssistantProvider(this._uML.getRootProfile(umlProfile)).getOwnedFilters().add(it);
  }

  private ProfileApplied toAppliedFilter(final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlProfile);
    final ProfileApplied _result;
    synchronized (_createCache_toAppliedFilter) {
      if (_createCache_toAppliedFilter.containsKey(_cacheKey)) {
        return _createCache_toAppliedFilter.get(_cacheKey);
      }
      ProfileApplied _createProfileApplied = FiltersUtil.umlFiltersFactory.createProfileApplied();
      _result = _createProfileApplied;
      _createCache_toAppliedFilter.put(_cacheKey, _result);
    }
    _init_toAppliedFilter(_result, umlProfile);
    return _result;
  }

  private final HashMap<ArrayList<?>, ProfileApplied> _createCache_toAppliedFilter = CollectionLiterals.newHashMap();

  private void _init_toAppliedFilter(final ProfileApplied it, final Profile umlProfile) {
    it.setProfileURI(EcoreUtil.getURI(umlProfile).toString());
    it.setProfileQualifiedName(umlProfile.getQualifiedName());
    String _qualifiedName = umlProfile.getQualifiedName();
    String _plus = (_qualifiedName + " is applied in context");
    it.setName(_plus);
  }

  protected Filter _andProfileApplied(final Void filter, final Profile umlProfile) {
    return null;
  }

  protected Filter _andProfileApplied(final Filter filter, final Profile umlProfile) {
    CompoundFilter _filter = this.toFilter(umlProfile);
    return this.operator_and(_filter, filter);
  }

  private AssistedElementTypeFilter toAssistedElementTypeFilter(final Profile umlProfile) {
    final ArrayList<?> _cacheKey = CollectionLiterals.newArrayList(umlProfile);
    final AssistedElementTypeFilter _result;
    synchronized (_createCache_toAssistedElementTypeFilter) {
      if (_createCache_toAssistedElementTypeFilter.containsKey(_cacheKey)) {
        return _createCache_toAssistedElementTypeFilter.get(_cacheKey);
      }
      AssistedElementTypeFilter _createAssistedElementTypeFilter = FiltersUtil.assistantFactory.createAssistedElementTypeFilter();
      _result = _createAssistedElementTypeFilter;
      _createCache_toAssistedElementTypeFilter.put(_cacheKey, _result);
    }
    _init_toAssistedElementTypeFilter(_result, umlProfile);
    return _result;
  }

  private final HashMap<ArrayList<?>, AssistedElementTypeFilter> _createCache_toAssistedElementTypeFilter = CollectionLiterals.newHashMap();

  private void _init_toAssistedElementTypeFilter(final AssistedElementTypeFilter it, final Profile umlProfile) {
  }

  protected boolean _isCompound(final Void filter) {
    return false;
  }

  protected boolean _isCompound(final Filter filter) {
    return false;
  }

  protected boolean _isCompound(final CompoundFilter filter) {
    return true;
  }

  protected Filter _reduce(final Void filter) {
    return null;
  }

  protected Filter _reduce(final Filter filter) {
    return filter;
  }

  protected Filter _reduce(final CompoundFilter filter) {
    Filter _switchResult = null;
    int _size = filter.getFilters().size();
    switch (_size) {
      case 0:
        _switchResult = null;
        break;
      case 1:
        _switchResult = filter.getFilters().get(0);
        break;
      default:
        _switchResult = filter;
        break;
    }
    return _switchResult;
  }

  private boolean add(final CompoundFilter compound, final Filter other) {
    boolean _xifexpression = false;
    EObject _eContainer = other.eContainer();
    boolean _notEquals = (!Objects.equals(_eContainer, null));
    if (_notEquals) {
      _xifexpression = compound.getFilters().add(other);
    } else {
      _xifexpression = compound.getOwnedFilters().add(other);
    }
    return _xifexpression;
  }

  private Boolean addAll(final CompoundFilter compound, final CompoundFilter other) {
    boolean _xifexpression = false;
    EObject _eContainer = other.eContainer();
    boolean _notEquals = (!Objects.equals(_eContainer, null));
    if (_notEquals) {
      final Consumer<Filter> _function = (Filter it) -> {
        this.add(compound, it);
      };
      other.getFilters().forEach(_function);
    } else {
      boolean _xblockexpression = false;
      {
        compound.getOwnedFilters().addAll(other.getOwnedFilters());
        _xblockexpression = compound.getFilters().addAll(other.getFilters());
      }
      _xifexpression = _xblockexpression;
    }
    return Boolean.valueOf(_xifexpression);
  }

  protected Filter _operator_or(final Void left, final Filter right) {
    return right;
  }

  protected Filter _operator_or(final Filter left, final Filter right) {
    return this.createOr(left, right);
  }

  private CompoundFilter createOr(final Filter left, final Filter right) {
    CompoundFilter _createCompoundFilter = FiltersUtil.filtersFactory.createCompoundFilter();
    final Procedure1<CompoundFilter> _function = (CompoundFilter it) -> {
      it.setOperator(OperatorKind.OR);
      this.add(it, left);
      this.add(it, right);
    };
    return ObjectExtensions.<CompoundFilter>operator_doubleArrow(_createCompoundFilter, _function);
  }

  protected Filter _operator_or(final CompoundFilter left, final Filter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = left.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case OR:
          CompoundFilter _xblockexpression = null;
          {
            this.add(left, right);
            _xblockexpression = left;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createOr(left, right);
          break;
      }
    } else {
      _switchResult = this.createOr(left, right);
    }
    return _switchResult;
  }

  protected Filter _operator_or(final Filter left, final CompoundFilter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = right.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case OR:
          CompoundFilter _xblockexpression = null;
          {
            this.add(right, left);
            _xblockexpression = right;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createOr(left, right);
          break;
      }
    } else {
      _switchResult = this.createOr(left, right);
    }
    return _switchResult;
  }

  protected Filter _operator_or(final CompoundFilter left, final CompoundFilter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = left.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case OR:
          CompoundFilter _xblockexpression = null;
          {
            OperatorKind _operator_1 = right.getOperator();
            if (_operator_1 != null) {
              switch (_operator_1) {
                case OR:
                  EObject _eContainer = right.eContainer();
                  boolean _equals = Objects.equals(_eContainer, null);
                  if (_equals) {
                    this.addAll(left, right);
                  } else {
                    this.add(left, right);
                  }
                  break;
                default:
                  this.add(left, right);
                  break;
              }
            } else {
              this.add(left, right);
            }
            _xblockexpression = left;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createOr(left, right);
          break;
      }
    } else {
      _switchResult = this.createOr(left, right);
    }
    return _switchResult;
  }

  protected Filter _operator_and(final Void left, final Filter right) {
    return right;
  }

  protected Filter _operator_and(final Filter left, final Filter right) {
    return this.createAnd(left, right);
  }

  private CompoundFilter createAnd(final Filter left, final Filter right) {
    CompoundFilter _createCompoundFilter = FiltersUtil.filtersFactory.createCompoundFilter();
    final Procedure1<CompoundFilter> _function = (CompoundFilter it) -> {
      it.setOperator(OperatorKind.AND);
      this.add(it, left);
      this.add(it, right);
    };
    return ObjectExtensions.<CompoundFilter>operator_doubleArrow(_createCompoundFilter, _function);
  }

  protected Filter _operator_and(final CompoundFilter left, final Filter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = left.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case AND:
          CompoundFilter _xblockexpression = null;
          {
            this.add(left, right);
            _xblockexpression = left;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createAnd(left, right);
          break;
      }
    } else {
      _switchResult = this.createAnd(left, right);
    }
    return _switchResult;
  }

  protected Filter _operator_and(final Filter left, final CompoundFilter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = right.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case AND:
          CompoundFilter _xblockexpression = null;
          {
            this.add(right, left);
            _xblockexpression = right;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createAnd(left, right);
          break;
      }
    } else {
      _switchResult = this.createAnd(left, right);
    }
    return _switchResult;
  }

  protected Filter _operator_and(final CompoundFilter left, final CompoundFilter right) {
    CompoundFilter _switchResult = null;
    OperatorKind _operator = left.getOperator();
    if (_operator != null) {
      switch (_operator) {
        case AND:
          CompoundFilter _xblockexpression = null;
          {
            OperatorKind _operator_1 = right.getOperator();
            if (_operator_1 != null) {
              switch (_operator_1) {
                case AND:
                  EObject _eContainer = right.eContainer();
                  boolean _equals = Objects.equals(_eContainer, null);
                  if (_equals) {
                    this.addAll(left, right);
                  } else {
                    this.add(left, right);
                  }
                  break;
                default:
                  this.add(left, right);
                  break;
              }
            } else {
              this.add(left, right);
            }
            _xblockexpression = left;
          }
          _switchResult = _xblockexpression;
          break;
        default:
          _switchResult = this.createAnd(left, right);
          break;
      }
    } else {
      _switchResult = this.createAnd(left, right);
    }
    return _switchResult;
  }

  @XbaseGenerated
  public Filter andProfileApplied(final Filter filter, final Profile umlProfile) {
    if (filter != null) {
      return _andProfileApplied(filter, umlProfile);
    } else if (filter == null) {
      return _andProfileApplied((Void)null, umlProfile);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(filter, umlProfile).toString());
    }
  }

  @XbaseGenerated
  public boolean isCompound(final Filter filter) {
    if (filter instanceof CompoundFilter) {
      return _isCompound((CompoundFilter)filter);
    } else if (filter != null) {
      return _isCompound(filter);
    } else {
      return _isCompound((Void)null);
    }
  }

  @XbaseGenerated
  public Filter reduce(final Filter filter) {
    if (filter instanceof CompoundFilter) {
      return _reduce((CompoundFilter)filter);
    } else if (filter != null) {
      return _reduce(filter);
    } else {
      return _reduce((Void)null);
    }
  }

  @XbaseGenerated
  public Filter operator_or(final Filter left, final Filter right) {
    if (left instanceof CompoundFilter
         && right instanceof CompoundFilter) {
      return _operator_or((CompoundFilter)left, (CompoundFilter)right);
    } else if (left instanceof CompoundFilter
         && right != null) {
      return _operator_or((CompoundFilter)left, right);
    } else if (left != null
         && right instanceof CompoundFilter) {
      return _operator_or(left, (CompoundFilter)right);
    } else if (left != null
         && right != null) {
      return _operator_or(left, right);
    } else if (left == null
         && right != null) {
      return _operator_or((Void)null, right);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(left, right).toString());
    }
  }

  @XbaseGenerated
  public Filter operator_and(final Filter left, final Filter right) {
    if (left instanceof CompoundFilter
         && right instanceof CompoundFilter) {
      return _operator_and((CompoundFilter)left, (CompoundFilter)right);
    } else if (left instanceof CompoundFilter
         && right != null) {
      return _operator_and((CompoundFilter)left, right);
    } else if (left != null
         && right instanceof CompoundFilter) {
      return _operator_and(left, (CompoundFilter)right);
    } else if (left != null
         && right != null) {
      return _operator_and(left, right);
    } else if (left == null
         && right != null) {
      return _operator_and((Void)null, right);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(left, right).toString());
    }
  }
}
