"use strict";
/********************************************************************************
 * Copyright (c) 2021 STMicroelectronics and others.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License 2.0 which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the Eclipse
 * Public License v. 2.0 are satisfied: GNU General Public License, version 2
 * with the GNU Classpath Exception which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 *******************************************************************************/
Object.defineProperty(exports, "__esModule", { value: true });
const clangd_config_1 = require("./clangd-config");
const chai_1 = require("chai");
const fs = require("fs");
const common_spec_1 = require("./common.spec");
const assert_1 = require("assert");
/* eslint-disable no-unused-expressions */
chai_1.use(require('chai-string'));
describe('clangd-config', function () {
    this.beforeAll(common_spec_1.createTmpDir);
    describe('load', function () {
        let clangdFile;
        this.beforeEach(function () {
            clangdFile = common_spec_1.tmpFile('.clangd', `CompileFlags:
    CompilationDatabase: >-
        /home/me/clangd-workspace/app/Debug_x86-64
`);
        });
        it('load a simple config file', function () {
            const simpleContent = clangd_config_1.load(clangdFile);
            chai_1.expect(simpleContent === null || simpleContent === void 0 ? void 0 : simpleContent.filePath).to.equal(clangdFile);
            chai_1.expect(simpleContent === null || simpleContent === void 0 ? void 0 : simpleContent.CompileFlags.CompilationDatabase).to.equal('/home/me/clangd-workspace/app/Debug_x86-64');
            chai_1.expect(simpleContent === null || simpleContent === void 0 ? void 0 : simpleContent.CompileFlags.Add).to.be.undefined;
            chai_1.expect(simpleContent === null || simpleContent === void 0 ? void 0 : simpleContent.CompileFlags.Remove).to.be.undefined;
        });
        it('load a non-existent config file', function () {
            const nonExistent = clangd_config_1.load('no/such/.clangd');
            chai_1.expect(nonExistent).to.be.undefined;
        });
        it('loadOrCreate an existing file', function () {
            const simpleContent = clangd_config_1.loadOrCreate(clangdFile);
            chai_1.expect(simpleContent.filePath).to.equal(clangdFile);
            chai_1.expect(simpleContent.CompileFlags.CompilationDatabase).to.equal('/home/me/clangd-workspace/app/Debug_x86-64');
            chai_1.expect(simpleContent.CompileFlags.Add).to.be.undefined;
            chai_1.expect(simpleContent.CompileFlags.Remove).to.be.undefined;
        });
        it('loadOrCreate a non-existent existing file', function () {
            const simpleContent = clangd_config_1.loadOrCreate('no/such/.clangd');
            chai_1.expect(simpleContent.filePath).to.have.string('no/such/.clangd');
            chai_1.expect(simpleContent.CompileFlags.CompilationDatabase).to.be.undefined;
            chai_1.expect(simpleContent.CompileFlags.Add).to.be.undefined;
            chai_1.expect(simpleContent.CompileFlags.Remove).to.be.undefined;
        });
        it('load a malformed YAML file', function () {
            const invalidClangdFile = common_spec_1.tmpFile('.clangd', 'This: \'is not valid YAML content.');
            try {
                const invalidContent = clangd_config_1.load(invalidClangdFile);
                assert_1.fail(`Should have thrown ClangdConfigException instead of getting ${invalidContent}`);
            }
            catch (e) {
                chai_1.expect(e).to.be.an.instanceOf(clangd_config_1.ClangdConfigException);
                const ex = e;
                chai_1.expect(ex.cause).to.exist;
            }
        });
    });
    describe('save', function () {
        let clangdFile;
        this.beforeEach(function () {
            clangdFile = common_spec_1.tmpFile('.clangd');
        });
        it('create a simple config file', function () {
            const config = {
                filePath: clangdFile,
                CompileFlags: {
                    CompilationDatabase: '/home/me/clangd-workspace/app/Release_x86-64'
                }
            };
            clangd_config_1.save(config);
            const simpleContent = fs.readFileSync(clangdFile, 'utf8');
            chai_1.expect(simpleContent).to.have.string('CompilationDatabase:');
            chai_1.expect(simpleContent).to.have.string('Release_x86-64');
            chai_1.expect(simpleContent).not.to.have.string('mocha', 'SourceFile path was stored');
        });
    });
});
//# sourceMappingURL=clangd-config.spec.js.map